(() => {
  "use strict";

  // ===== Storage keys =====
  const USERS_KEY = "stamp_users_v1";
  const COLLECTIONS_KEY = "collections_v1";
  const SESSION_KEY = "stamps_session_v1";
  const CATALOG_KEY_PREFIX = "catalog_v1:"; // catalog_v1:<collectionId>
  const STATE_KEY_PREFIX = "state_v1:";     // state_v1:<username>:<collectionId>

  // Rarity order for sorting
  const RARITY_ORDER = { common: 1, uncommon: 2, rare: 3, epic: 4, legendary: 5 };

  // ===== DOM: Login =====
  const loginSection = document.getElementById("login-section");
  const loginForm = document.getElementById("login-form");
  const loginUsername = document.getElementById("login-username");
  const loginPassword = document.getElementById("login-password");
  const loginError = document.getElementById("login-error");

  // ===== DOM: App shell =====
  const app = document.getElementById("app");
  const appTitle = document.getElementById("app-title");
  const currentUserLabel = document.getElementById("current-user-label");
  const btnLogout = document.getElementById("btn-logout");

  // Sidebar
  const sidebarNav = document.getElementById("sidebar-nav");
  const sidebarUser = document.getElementById("sidebar-user");

  // ===== DOM: Admin controls =====
  const adminControls = document.getElementById("admin-controls");
  const btnManageStamps = document.getElementById("btn-manage-stamps");
  const btnManageCollections = document.getElementById("btn-manage-collections");
  const btnManageUsers = document.getElementById("btn-manage-users");

  // ===== DOM: Filters / Grid / Stats =====
  const grid = document.getElementById("grid");
  const search = document.getElementById("search");
  const filterSet = document.getElementById("filter-set");
  const filterRarity = document.getElementById("filter-rarity");
  const sortBy = document.getElementById("sort-by");
  const ownedOnly = document.getElementById("show-owned-only");
  const wishlistOnly = document.getElementById("show-wishlist-only");
  const exportBtn = document.getElementById("export-btn");
  const importBtn = document.getElementById("import-btn");
  const importFile = document.getElementById("import-file");
  const resetBtn = document.getElementById("reset-btn");
  const ownedCountEl = document.getElementById("owned-count");
  const totalCountEl = document.getElementById("total-count");
  const progressBar = document.getElementById("progress-bar");

  // ===== DOM: Modals - Stamps =====
  const modalStamps = document.getElementById("modal-stamps");
  const closeStamps = document.getElementById("close-stamps");
  const openStampForm = document.getElementById("open-stamp-form");
  const stampsTableBody = document.getElementById("stamps-table-body");
  const stampsModalTitle = document.getElementById("stamps-modal-title");

  // ===== DOM: Modal - Stamp form =====
  const modalStampForm = document.getElementById("modal-stamp-form");
  const closeStampForm = document.getElementById("close-stamp-form");
  const stampForm = document.getElementById("stamp-form");
  const stampFormTitle = document.getElementById("stamp-form-title");
  const stampId = document.getElementById("stamp-id");
  const stampName = document.getElementById("stamp-name");
  const stampSet = document.getElementById("stamp-set");
  const stampNumber = document.getElementById("stamp-number");
  const stampRarity = document.getElementById("stamp-rarity");
  const stampImage = document.getElementById("stamp-image");

  // ===== DOM: Modals - Users =====
  const modalUsers = document.getElementById("modal-users");
  const closeUsers = document.getElementById("close-users");
  const userForm = document.getElementById("user-form");
  const userId = document.getElementById("user-id");
  const userUsername = document.getElementById("user-username");
  const userPassword = document.getElementById("user-password");
  const userRole = document.getElementById("user-role");
  const userResetForm = document.getElementById("user-reset-form");
  const usersTableBody = document.getElementById("users-table-body");

  // ===== DOM: Modals - Collections =====
  const modalCollections = document.getElementById("modal-collections");
  const closeCollections = document.getElementById("close-collections");
  const collectionForm = document.getElementById("collection-form");
  const collectionId = document.getElementById("collection-id");
  const collectionName = document.getElementById("collection-name");
  const collectionIcon = document.getElementById("collection-icon");
  const collectionResetForm = document.getElementById("collection-reset-form");
  const collectionsTableBody = document.getElementById("collections-table-body");

  // ===== Data: Users =====
  let users = [];
  function loadUsers() {
    try {
      const raw = localStorage.getItem(USERS_KEY);
      users = raw ? JSON.parse(raw) : [];
      if (!Array.isArray(users)) users = [];
    } catch (_) {
      users = [];
    }
  }
  function saveUsers() {
    localStorage.setItem(USERS_KEY, JSON.stringify(users));
  }
  function ensureDefaultAdmin() {
    loadUsers();
    if (!users.some((u) => u.role === "admin")) {
      users.push({ id: 1, username: "admin", password: "admin", role: "admin" });
      saveUsers();
    }
  }

  // ===== Data: Collections =====
  let collections = [];
  let currentCollectionId = "stamps";

  // ===== Simple hash router (each collection has its own URL) =====
  // Routes:
  //   #/collection/<collectionId>
  //   #/dashboard  (reserved for future)
  function setRoute(route) {
    const normalized = String(route || "").replace(/^#?\/?/, "");
    const next = "#/" + normalized;
    if (location.hash !== next) location.hash = next;
  }

  function parseRoute() {
    const raw = String(location.hash || "").replace(/^#\/?/, "");
    const parts = raw.split("/").filter(Boolean);
    if (parts.length === 0) return { name: "collection", collectionId: currentCollectionId };
    if (parts[0] === "dashboard") return { name: "dashboard" };
    if (parts[0] === "collection") return { name: "collection", collectionId: parts[1] || currentCollectionId };
    // Backward compatibility: allow #/<collectionId>
    return { name: "collection", collectionId: parts[0] || currentCollectionId };
  }

  function router() {
    const r = parseRoute();
    if (r.name === "collection") {
      const target = String(r.collectionId || currentCollectionId);
      if (target !== currentCollectionId) {
        switchCollection(target, { syncRoute: false });
      } else {
        // Ensure UI reflects current
        renderSidebar();
        updateAppTitles();
      }
      return;
    }
    // dashboard reserved; fall back to current collection
    setRoute(`collection/${encodeURIComponent(currentCollectionId)}`);
  }

  function defaultCollections() {
    return [
      { id: "stamps", name: "Známky", icon: "📮" },
      { id: "coins", name: "Mince", icon: "🪙" },
      { id: "cards", name: "Karty", icon: "🃏" }
    ];
  }

  function loadCollections() {
    try {
      const raw = localStorage.getItem(COLLECTIONS_KEY);
      const parsed = raw ? JSON.parse(raw) : null;
      collections = Array.isArray(parsed) ? parsed : [];
    } catch (_) {
      collections = [];
    }

    if (!Array.isArray(collections) || collections.length === 0) {
      collections = defaultCollections();
      saveCollections();
    }

    // Ensure essential fields
    collections = collections
      .filter((c) => c && typeof c === "object")
      .map((c) => ({
        id: String(c.id || "").trim() || "col_" + Math.random().toString(16).slice(2),
        name: String(c.name || "Sbírka").trim() || "Sbírka",
        icon: String(c.icon || "📦").trim() || "📦"
      }));

    // Ensure unique IDs
    const seen = new Set();
    collections.forEach((c) => {
      let base = c.id;
      let i = 2;
      while (seen.has(c.id)) {
        c.id = base + "_" + i;
        i++;
      }
      seen.add(c.id);
    });

    saveCollections();
  }

  function saveCollections() {
    localStorage.setItem(COLLECTIONS_KEY, JSON.stringify(collections));
  }

  function getCollection(id) {
    return collections.find((c) => c.id === id) || null;
  }

  function normalizeCollectionId(str) {
    const s = String(str || "").trim().toLowerCase();
    const cleaned = s
      .normalize("NFKD")
      .replace(/[\u0300-\u036f]/g, "")
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/^-+|-+$/g, "")
      .slice(0, 32);
    return cleaned || "collection";
  }

  // ===== Data: Catalog (per collection) =====
  let catalog = [];

  function catalogKey(id) {
    return CATALOG_KEY_PREFIX + id;
  }

  function buildDefaultCatalogFor(collectionId) {
    // Keep it simple: stamps have the original sample catalog.
    if (collectionId === "stamps") {
      const sets = ["Klasická edice", "Příroda", "Památky"];
      const names = {
        "Klasická edice": ["Poštovní trubka", "Hradčany", "Holubice", "Dopisy", "Filatelie", "Sbírka", "Katalog", "Album"],
        "Příroda": ["Zubr", "Rys", "Orel", "Jelen", "Vlk", "Sova", "Ledňáček", "Medvěd"],
        "Památky": ["Karlštejn", "Orloj", "Karlův most", "Špilberk", "Kroměříž", "Lednice", "Hluboká", "Bouzov"]
      };
      const rarCycle = ["common", "uncommon", "rare", "epic", "legendary", "common", "uncommon", "rare"];
      let id = 1;
      const out = [];
      for (const set of sets) {
        const list = names[set];
        list.forEach((n, i) => {
          const rarity = rarCycle[i % rarCycle.length];
          const number = i + 1;
          out.push({
            id: id++,
            name: n,
            set,
            number,
            rarity,
            image: `https://picsum.photos/seed/zn-${set.replace(/\s/g, "_")}-${number}/400/550`
          });
        });
      }
      return out;
    }

    // Coins / cards: small placeholder sample
    if (collectionId === "coins") {
      return [
        { id: 1, name: "Koruna 1912", set: "Historické", number: 1, rarity: "rare", image: "" },
        { id: 2, name: "20 Kč 1994", set: "Česká republika", number: 2, rarity: "uncommon", image: "" },
        { id: 3, name: "Dukát", set: "Zlaté", number: 3, rarity: "legendary", image: "" }
      ];
    }

    if (collectionId === "cards") {
      return [
        { id: 1, name: "Drak", set: "Základní", number: 1, rarity: "epic", image: "" },
        { id: 2, name: "Rytíř", set: "Základní", number: 2, rarity: "rare", image: "" },
        { id: 3, name: "Léčitel", set: "Rozšíření", number: 3, rarity: "uncommon", image: "" }
      ];
    }

    return [];
  }

  function loadCatalogFor(collectionId) {
    try {
      const raw = localStorage.getItem(catalogKey(collectionId));
      const parsed = raw ? JSON.parse(raw) : null;
      catalog = Array.isArray(parsed) ? parsed : [];
    } catch (_) {
      catalog = [];
    }

    if (!Array.isArray(catalog) || catalog.length === 0) {
      catalog = buildDefaultCatalogFor(collectionId);
      saveCatalogFor(collectionId);
    }

    // Basic sanitization
    catalog = catalog
      .filter((x) => x && typeof x === "object")
      .map((x) => ({
        id: Number(x.id),
        name: String(x.name || "").trim(),
        set: String(x.set || "").trim(),
        number: Number(x.number || 0),
        rarity: ["common", "uncommon", "rare", "epic", "legendary"].includes(String(x.rarity)) ? String(x.rarity) : "common",
        image: String(x.image || "").trim()
      }))
      .filter((x) => x.id > 0 && x.name && x.set && x.number > 0);

    // Ensure unique ids
    const used = new Set();
    catalog.forEach((it) => {
      if (!used.has(it.id)) {
        used.add(it.id);
        return;
      }
      let next = Math.max(...Array.from(used)) + 1;
      while (used.has(next)) next++;
      it.id = next;
      used.add(next);
    });

    saveCatalogFor(collectionId);
  }

  function saveCatalogFor(collectionId) {
    localStorage.setItem(catalogKey(collectionId), JSON.stringify(catalog));
  }

  function nextCatalogId() {
    return (catalog.reduce((m, it) => Math.max(m, it.id), 0) + 1) || 1;
  }

  // ===== Data: Per-user collection state (per collection) =====
  let currentUser = null;
  let state = { owned: {}, dupes: {}, wishlist: {} };

  function stateKey(username, collectionId) {
    const name = username || "guest";
    const col = collectionId || "stamps";
    return `${STATE_KEY_PREFIX}${name}:${col}`;
  }

  function loadStateFor(username, collectionId) {
    try {
      const raw = localStorage.getItem(stateKey(username, collectionId));
      const parsed = raw ? JSON.parse(raw) : null;
      if (parsed && typeof parsed === "object") {
        state.owned = parsed.owned || {};
        state.dupes = parsed.dupes || {};
        state.wishlist = parsed.wishlist || {};
      } else {
        state = { owned: {}, dupes: {}, wishlist: {} };
      }
    } catch (_) {
      state = { owned: {}, dupes: {}, wishlist: {} };
    }
  }

  function saveStateFor(username, collectionId) {
    localStorage.setItem(stateKey(username, collectionId), JSON.stringify(state));
  }

  // ===== Session =====
  function persistSession() {
    if (!currentUser) return;
    localStorage.setItem(
      SESSION_KEY,
      JSON.stringify({ username: currentUser.username, collectionId: currentCollectionId })
    );
  }

  function clearSession() {
    localStorage.removeItem(SESSION_KEY);
  }

  function tryAutoLogin() {
    try {
      const raw = localStorage.getItem(SESSION_KEY);
      if (!raw) return false;
      const sess = JSON.parse(raw);
      if (!sess || !sess.username) return false;
      const u = users.find((x) => x.username === sess.username);
      if (!u) return false;
      currentUser = { id: u.id, username: u.username, role: u.role };
      if (sess.collectionId) currentCollectionId = String(sess.collectionId);
      return true;
    } catch (_) {
      return false;
    }
  }

  // ===== Helpers =====
  const isOwned = (id) => !!state.owned[id];
  const dupeCount = (id) => Number(state.dupes[id] || 0);
  const inWishlist = (id) => !!state.wishlist[id];

  function updateStats() {
    const total = catalog.length;
    const owned = catalog.reduce((acc, s) => acc + (isOwned(s.id) ? 1 : 0), 0);
    ownedCountEl.textContent = String(owned);
    totalCountEl.textContent = String(total);
    const pct = total ? Math.round((owned / total) * 100) : 0;
    progressBar.style.width = pct + "%";
    progressBar.title = pct + "%";

    // Update sidebar counts in "real time" when ownership changes
    updateSidebarCounts();
  }

  function updateSidebarCounts() {
    if (!sidebarNav) return;
    const buttons = sidebarNav.querySelectorAll(".sidebar-item");
    buttons.forEach((btn) => {
      const colId = btn.dataset.collection;
      if (!colId) return;

      // For the currently active collection, we can use the in-memory catalog/state.
      // For the others, we compute counts from localStorage.
      let owned = 0;
      let total = 0;
      if (colId === currentCollectionId) {
        total = catalog.length;
        owned = catalog.reduce((acc, s) => acc + (isOwned(s.id) ? 1 : 0), 0);
      } else {
        owned = getOwnedCountForCollection(colId);
        total = getTotalCountForCollection(colId);
      }

      const countsEl = btn.querySelector(".counts");
      if (countsEl) countsEl.textContent = `Nasbíráno: ${owned}/${total}`;

      const pillEl = btn.querySelector(".pill");
      if (pillEl) pillEl.textContent = total ? `${Math.round((owned / total) * 100)}%` : "0%";
    });
  }

  function uniqueSets() {
    return Array.from(new Set(catalog.map((s) => s.set)));
  }

  function populateSetFilter() {
    filterSet.innerHTML = "";
    const allOpt = document.createElement("option");
    allOpt.value = "all";
    allOpt.textContent = "Všechny série";
    filterSet.appendChild(allOpt);

    for (const s of uniqueSets()) {
      const opt = document.createElement("option");
      opt.value = s;
      opt.textContent = s;
      filterSet.appendChild(opt);
    }
    filterSet.value = "all";
  }

  function compareBy(a, b, key, dir = "asc") {
    let va = a[key];
    let vb = b[key];
    if (key === "rarity") {
      va = RARITY_ORDER[va] || 0;
      vb = RARITY_ORDER[vb] || 0;
    }
    if (typeof va === "string") va = va.toLowerCase();
    if (typeof vb === "string") vb = vb.toLowerCase();
    const cmp = va < vb ? -1 : va > vb ? 1 : 0;
    return dir === "asc" ? cmp : -cmp;
  }

  function sortItems(items) {
    const mode = sortBy.value;
    const copy = items.slice();
    switch (mode) {
      case "name-asc":
        return copy.sort((a, b) => compareBy(a, b, "name", "asc"));
      case "name-desc":
        return copy.sort((a, b) => compareBy(a, b, "name", "desc"));
      case "number-asc":
        return copy.sort((a, b) => compareBy(a, b, "number", "asc"));
      case "number-desc":
        return copy.sort((a, b) => compareBy(a, b, "number", "desc"));
      case "rarity-desc":
        return copy.sort((a, b) => compareBy(a, b, "rarity", "desc"));
      case "rarity-asc":
        return copy.sort((a, b) => compareBy(a, b, "rarity", "asc"));
      default:
        return copy;
    }
  }

  function applyFiltersAndRender() {
    const q = (search.value || "").trim().toLowerCase();
    const setVal = filterSet.value;
    const rarVal = filterRarity.value;
    const wantOwned = ownedOnly.checked;
    const wantWishlist = wishlistOnly.checked;

    let items = catalog.filter((s) => {
      if (q) {
        const inText = s.name.toLowerCase().includes(q) || s.set.toLowerCase().includes(q);
        if (!inText) return false;
      }
      if (setVal !== "all" && s.set !== setVal) return false;
      if (rarVal !== "all" && s.rarity !== rarVal) return false;
      if (wantOwned && !isOwned(s.id)) return false;
      if (wantWishlist && !inWishlist(s.id)) return false;
      return true;
    });

    items = sortItems(items);
    renderGrid(items);
    updateStats();
    renderSidebar();
  }

  function createEl(tag, cls) {
    const el = document.createElement(tag);
    if (cls) el.className = cls;
    return el;
  }

  function createCard(item) {
    const card = createEl("div", "card clickable");
    card.setAttribute("data-id", String(item.id));
    card.title = "Klikněte pro přepnutí vlastnění";

    const thumb = createEl("div", "card-thumb");
    const img = document.createElement("img");
    img.src = item.image || `https://picsum.photos/seed/zn-${currentCollectionId}-${item.id}/400/550`;
    img.alt = item.name;
    thumb.appendChild(img);
    card.appendChild(thumb);

    if (isOwned(item.id)) {
      const flag = createEl("div", "owned-flag");
      flag.textContent = "Vlastním";
      card.appendChild(flag);
    }

    const dupes = dupeCount(item.id);
    if (dupes > 0) {
      const d = createEl("div", "dupe-badge");
      d.textContent = `Duplikáty: ${dupes}`;
      card.appendChild(d);
    }

    const body = createEl("div", "card-body");
    const title = createEl("div", "card-title");
    title.textContent = item.name;
    const sub = createEl("div", "card-sub");
    sub.textContent = item.set;
    const badges = createEl("div", "badges");
    const bNumber = createEl("div", "badge number");
    bNumber.textContent = "#" + String(item.number).padStart(3, "0");
    const bRarity = createEl("div", "badge rarity-" + item.rarity);
    bRarity.textContent = item.rarity.toUpperCase();
    badges.appendChild(bNumber);
    badges.appendChild(bRarity);
    body.appendChild(title);
    body.appendChild(sub);
    body.appendChild(badges);
    card.appendChild(body);

    const actions = createEl("div", "card-actions");
    const wish = createEl("button", "btn small wishlist");
    wish.innerHTML = (inWishlist(item.id) ? "⭐" : "☆") + " Wishlist";
    if (inWishlist(item.id)) wish.classList.add("active");
    const counter = createEl("div", "counter");
    const minus = createEl("button", "btn small");
    minus.textContent = "−";
    const qty = createEl("span", "qty");
    qty.textContent = String(dupes);
    const plus = createEl("button", "btn small");
    plus.textContent = "+";
    counter.appendChild(minus);
    counter.appendChild(qty);
    counter.appendChild(plus);
    actions.appendChild(wish);
    actions.appendChild(counter);
    card.appendChild(actions);

    // Events
    card.addEventListener("click", () => {
      const id = item.id;
      if (isOwned(id)) {
        delete state.owned[id];
        state.dupes[id] = 0;
      } else {
        state.owned[id] = true;
      }
      saveStateFor(currentUser?.username, currentCollectionId);
      applyFiltersAndRender();
    });

    function stop(e) {
      e.stopPropagation();
    }

    wish.addEventListener("click", (e) => {
      stop(e);
      const id = item.id;
      if (inWishlist(id)) {
        delete state.wishlist[id];
      } else {
        state.wishlist[id] = true;
      }
      saveStateFor(currentUser?.username, currentCollectionId);
      applyFiltersAndRender();
    });

    plus.addEventListener("click", (e) => {
      stop(e);
      const id = item.id;
      const next = dupeCount(id) + 1;
      state.dupes[id] = next;
      if (next > 0 && !isOwned(id)) state.owned[id] = true;
      saveStateFor(currentUser?.username, currentCollectionId);
      applyFiltersAndRender();
    });

    minus.addEventListener("click", (e) => {
      stop(e);
      const id = item.id;
      const curr = dupeCount(id);
      const next = Math.max(0, curr - 1);
      state.dupes[id] = next;
      saveStateFor(currentUser?.username, currentCollectionId);
      applyFiltersAndRender();
    });

    return card;
  }

  function renderGrid(items) {
    grid.innerHTML = "";
    if (items.length === 0) {
      const empty = createEl("div");
      empty.style.color = "#a9afc7";
      empty.style.padding = "20px";
      empty.textContent = "Žádné karty neodpovídají filtru.";
      grid.appendChild(empty);
      return;
    }
    for (const it of items) {
      grid.appendChild(createCard(it));
    }
  }

  // ===== Sidebar rendering =====
  function getOwnedCountForCollection(colId) {
    const uname = currentUser?.username || "guest";
    try {
      const raw = localStorage.getItem(stateKey(uname, colId));
      if (!raw) return 0;
      const parsed = JSON.parse(raw);
      const owned = parsed && parsed.owned ? parsed.owned : {};
      if (!owned || typeof owned !== "object") return 0;
      return Object.keys(owned).length;
    } catch (_) {
      return 0;
    }
  }

  function getTotalCountForCollection(colId) {
    try {
      const raw = localStorage.getItem(catalogKey(colId));
      const parsed = raw ? JSON.parse(raw) : null;
      if (Array.isArray(parsed)) return parsed.length;
      // if no catalog yet, default would be created on first open; estimate via builder
      return buildDefaultCatalogFor(colId).length;
    } catch (_) {
      return buildDefaultCatalogFor(colId).length;
    }
  }

  function renderSidebar() {
    if (!sidebarNav) return;
    sidebarNav.innerHTML = "";

    const uname = currentUser?.username || "";
    sidebarUser.textContent = uname ? `Přihlášen: ${uname}` : "";

    for (const c of collections) {
      const btn = document.createElement("button");
      btn.type = "button";
      btn.className = "sidebar-item" + (c.id === currentCollectionId ? " active" : "");
      btn.dataset.collection = c.id;

      const icon = document.createElement("div");
      icon.className = "icon";
      icon.textContent = c.icon || "📦";

      const name = document.createElement("div");
      name.className = "name";
      name.textContent = c.name;

      const owned = getOwnedCountForCollection(c.id);
      const total = getTotalCountForCollection(c.id);

      const counts = document.createElement("div");
      counts.className = "counts";
      counts.textContent = `Nasbíráno: ${owned}/${total}`;

      const pill = document.createElement("div");
      pill.className = "pill";
      pill.textContent = total ? `${Math.round((owned / total) * 100)}%` : "0%";

      btn.appendChild(icon);
      btn.appendChild(name);
      btn.appendChild(counts);
      btn.appendChild(pill);

      btn.addEventListener("click", () => {
        // Route change will trigger router() and switch the content.
        setRoute(`collection/${encodeURIComponent(c.id)}`);
      });

      sidebarNav.appendChild(btn);
    }
  }

  function switchCollection(collectionId, opts = {}) {
    const { syncRoute = true } = opts;
    if (!getCollection(collectionId)) return;
    currentCollectionId = collectionId;
    persistSession();

    // Keep URL in sync (each collection has its own page)
    if (syncRoute) setRoute(`collection/${encodeURIComponent(currentCollectionId)}`);

    // Load catalog + state for selected collection
    loadCatalogFor(currentCollectionId);
    loadStateFor(currentUser?.username, currentCollectionId);

    // Reset UI filters to be safe
    search.value = "";
    filterRarity.value = "all";
    sortBy.value = "name-asc";
    ownedOnly.checked = false;
    wishlistOnly.checked = false;

    populateSetFilter();
    totalCountEl.textContent = String(catalog.length);
    updateAppTitles();
    applyFiltersAndRender();
  }

  function updateAppTitles() {
    const c = getCollection(currentCollectionId);
    const name = c ? c.name : "Sbírka";
    appTitle.textContent = `Album – ${name}`;
    stampsModalTitle.textContent = `Správa položek – ${name}`;
  }

  // ===== Events on filters =====
  search.addEventListener("input", applyFiltersAndRender);
  filterSet.addEventListener("change", applyFiltersAndRender);
  filterRarity.addEventListener("change", applyFiltersAndRender);
  sortBy.addEventListener("change", applyFiltersAndRender);
  ownedOnly.addEventListener("change", applyFiltersAndRender);
  wishlistOnly.addEventListener("change", applyFiltersAndRender);

  // ===== Export / Import / Reset (per user + per collection) =====
  exportBtn.addEventListener("click", () => {
    const payload = {
      version: 2,
      exportedAt: new Date().toISOString(),
      username: currentUser?.username || null,
      collectionId: currentCollectionId,
      state
    };
    const blob = new Blob([JSON.stringify(payload, null, 2)], { type: "application/json" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `sbirka-${currentCollectionId}-export.json`;
    document.body.appendChild(a);
    a.click();
    a.remove();
    URL.revokeObjectURL(url);
  });

  importBtn.addEventListener("click", () => importFile.click());
  importFile.addEventListener("change", async (e) => {
    const file = e.target.files && e.target.files[0];
    if (!file) return;
    try {
      const text = await file.text();
      const data = JSON.parse(text);
      if (!data || typeof data !== "object" || !data.state) throw new Error("Neplatný soubor");

      // If file contains collectionId, switch to it (if exists)
      const col = data.collectionId ? String(data.collectionId) : currentCollectionId;
      if (getCollection(col)) {
        currentCollectionId = col;
        persistSession();
        loadCatalogFor(currentCollectionId);
      }

      const st = data.state;
      state.owned = st.owned || {};
      state.dupes = st.dupes || {};
      state.wishlist = st.wishlist || {};
      saveStateFor(currentUser?.username, currentCollectionId);
      populateSetFilter();
      applyFiltersAndRender();
      importFile.value = "";
    } catch (err) {
      alert("Import selhal: " + (err && err.message ? err.message : String(err)));
    }
  });

  resetBtn.addEventListener("click", () => {
    if (!confirm("Opravdu chcete vymazat stav vaší kolekce pro tuto sbírku?")) return;
    state = { owned: {}, dupes: {}, wishlist: {} };
    localStorage.removeItem(stateKey(currentUser?.username, currentCollectionId));
    applyFiltersAndRender();
  });

  // ===== Login / Logout =====
  function setAppVisibility(loggedIn) {
    loginSection.style.display = loggedIn ? "none" : "block";
    app.style.display = loggedIn ? "block" : "none";
    if (loggedIn) {
      currentUserLabel.textContent = `Přihlášen: ${currentUser.username} (${currentUser.role})`;
      adminControls.style.display = currentUser.role === "admin" ? "flex" : "none";
      updateAppTitles();
    } else {
      currentUserLabel.textContent = "";
      adminControls.style.display = "none";
    }
  }

  loginForm.addEventListener("submit", (e) => {
    e.preventDefault();
    loginError.textContent = "";
    const user = (loginUsername.value || "").trim();
    const pass = (loginPassword.value || "").trim();
    if (!user || !pass) {
      loginError.textContent = "Vyplňte uživatelské jméno a heslo.";
      return;
    }
    const found = users.find((u) => u.username === user);
    if (!found || (found.password || "") !== pass) {
      loginError.textContent = "Neplatné přihlášení.";
      return;
    }

    currentUser = { id: found.id, username: found.username, role: found.role };

    // Default to first existing collection if invalid
    if (!getCollection(currentCollectionId)) currentCollectionId = collections[0]?.id || "stamps";

    persistSession();
    setAppVisibility(true);

    // Default route if missing
    if (!String(location.hash || "").trim()) {
      setRoute(`collection/${encodeURIComponent(currentCollectionId)}`);
    }

    // Render current collection, then let router correct if URL points elsewhere
    switchCollection(currentCollectionId, { syncRoute: false });
    router();
  });

  btnLogout.addEventListener("click", () => {
    currentUser = null;
    clearSession();
    setAppVisibility(false);
    loginForm.reset();
  });

  // ===== Modals helpers =====
  function openModal(el) {
    el.style.display = "block";
  }
  function closeModal(el) {
    el.style.display = "none";
  }

  function requireAdmin() {
    if (!currentUser || currentUser.role !== "admin") {
      alert("Akce vyžaduje roli admin.");
      return false;
    }
    return true;
  }

  // ===== Admin: Manage Items (catalog) =====
  btnManageStamps.addEventListener("click", () => {
    if (!requireAdmin()) return;
    renderStampsTable();
    openModal(modalStamps);
  });

  closeStamps.addEventListener("click", () => closeModal(modalStamps));

  openStampForm.addEventListener("click", () => {
    if (!requireAdmin()) return;
    stampFormTitle.textContent = "Přidat položku";
    stampId.value = "";
    stampName.value = "";
    stampSet.value = "";
    stampNumber.value = "";
    stampRarity.value = "common";
    stampImage.value = "";
    openModal(modalStampForm);
  });

  closeStampForm.addEventListener("click", () => closeModal(modalStampForm));

  function renderStampsTable() {
    stampsTableBody.innerHTML = "";
    const rows = catalog
      .slice()
      .sort((a, b) => {
        const s = a.set.localeCompare(b.set);
        return s !== 0 ? s : a.number - b.number;
      });

    for (const it of rows) {
      const tr = document.createElement("tr");
      tr.innerHTML = `
        <td>${it.id}</td>
        <td>${it.name}</td>
        <td>${it.set}</td>
        <td>${it.number}</td>
        <td>${it.rarity}</td>
        <td>${it.image ? `<a href="${it.image}" target="_blank">odkaz</a>` : "-"}</td>
        <td>
          <button class="btn small" data-edit="${it.id}">Upravit</button>
          <button class="btn small danger" data-del="${it.id}">Smazat</button>
        </td>
      `;
      stampsTableBody.appendChild(tr);
    }

    stampsTableBody.querySelectorAll("[data-edit]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = Number(e.currentTarget.getAttribute("data-edit"));
        const it = catalog.find((x) => x.id === id);
        if (!it) return;
        stampFormTitle.textContent = "Upravit položku";
        stampId.value = String(it.id);
        stampName.value = it.name;
        stampSet.value = it.set;
        stampNumber.value = String(it.number);
        stampRarity.value = it.rarity;
        stampImage.value = it.image || "";
        openModal(modalStampForm);
      });
    });

    stampsTableBody.querySelectorAll("[data-del]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = Number(e.currentTarget.getAttribute("data-del"));
        if (!confirm("Opravdu smazat položku #" + id + "?")) return;
        catalog = catalog.filter((x) => x.id !== id);
        saveCatalogFor(currentCollectionId);

        // Clean current user's state for this id
        delete state.owned[id];
        delete state.dupes[id];
        delete state.wishlist[id];
        saveStateFor(currentUser?.username, currentCollectionId);

        populateSetFilter();
        applyFiltersAndRender();
        renderStampsTable();
      });
    });
  }

  stampForm.addEventListener("submit", (e) => {
    e.preventDefault();
    if (!requireAdmin()) return;

    const idVal = (stampId.value || "").trim();
    const nameVal = (stampName.value || "").trim();
    const setVal = (stampSet.value || "").trim();
    const numberVal = Number(stampNumber.value);
    const rarityVal = (stampRarity.value || "common").trim();
    const imageVal = (stampImage.value || "").trim();

    if (!nameVal || !setVal || !numberVal || numberVal < 1) {
      alert("Vyplňte název, sérii a kladné číslo.");
      return;
    }

    const payload = {
      name: nameVal,
      set: setVal,
      number: numberVal,
      rarity: ["common", "uncommon", "rare", "epic", "legendary"].includes(rarityVal) ? rarityVal : "common",
      image: imageVal || ""
    };

    if (idVal) {
      const id = Number(idVal);
      const idx = catalog.findIndex((x) => x.id === id);
      if (idx >= 0) catalog[idx] = { ...catalog[idx], ...payload };
    } else {
      const id = nextCatalogId();
      catalog.push({ id, ...payload });
    }

    saveCatalogFor(currentCollectionId);
    populateSetFilter();
    applyFiltersAndRender();
    renderStampsTable();
    closeModal(modalStampForm);
  });

  // ===== Admin: Manage Users =====
  btnManageUsers.addEventListener("click", () => {
    if (!requireAdmin()) return;
    userFormReset();
    renderUsersTable();
    openModal(modalUsers);
  });

  closeUsers.addEventListener("click", () => closeModal(modalUsers));

  function userFormReset() {
    userId.value = "";
    userUsername.value = "";
    userPassword.value = "";
    userRole.value = "user";
  }

  userResetForm.addEventListener("click", userFormReset);

  function renderUsersTable() {
    usersTableBody.innerHTML = "";
    const rows = users.slice().sort((a, b) => a.username.localeCompare(b.username));
    for (const u of rows) {
      const tr = document.createElement("tr");
      tr.innerHTML = `
        <td>${u.id}</td>
        <td>${u.username}</td>
        <td>${u.role}</td>
        <td>
          <button class="btn small" data-uedit="${u.id}">Upravit</button>
          <button class="btn small danger" data-udel="${u.id}">Smazat</button>
        </td>
      `;
      usersTableBody.appendChild(tr);
    }

    usersTableBody.querySelectorAll("[data-uedit]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = Number(e.currentTarget.getAttribute("data-uedit"));
        const u = users.find((x) => x.id === id);
        if (!u) return;
        userId.value = String(u.id);
        userUsername.value = u.username;
        userPassword.value = "";
        userRole.value = u.role;
      });
    });

    usersTableBody.querySelectorAll("[data-udel]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = Number(e.currentTarget.getAttribute("data-udel"));
        const u = users.find((x) => x.id === id);
        if (!u) return;
        if (!confirm(`Opravdu smazat uživatele "${u.username}"?`)) return;

        const admins = users.filter((x) => x.role === "admin");
        if (u.role === "admin" && admins.length <= 1) {
          alert("Nelze smazat posledního admina.");
          return;
        }
        if (currentUser && currentUser.id === u.id) {
          alert("Nemůžete smazat sám sebe během přihlášení.");
          return;
        }
        users = users.filter((x) => x.id !== id);
        saveUsers();
        renderUsersTable();
      });
    });
  }

  userForm.addEventListener("submit", (e) => {
    e.preventDefault();
    if (!requireAdmin()) return;

    const idVal = (userId.value || "").trim();
    const usernameVal = (userUsername.value || "").trim();
    const passwordVal = (userPassword.value || "").trim();
    const roleVal = (userRole.value || "user").trim();

    if (!usernameVal) {
      alert("Uživatelské jméno je povinné.");
      return;
    }

    const exists = users.find((u) => u.username === usernameVal && String(u.id) !== idVal);
    if (exists) {
      alert("Uživatel s tímto jménem již existuje.");
      return;
    }

    if (idVal) {
      const id = Number(idVal);
      const idx = users.findIndex((x) => x.id === id);
      if (idx >= 0) {
        users[idx].username = usernameVal;
        if (passwordVal) users[idx].password = passwordVal;
        users[idx].role = roleVal === "admin" ? "admin" : "user";

        if (currentUser && currentUser.id === id) {
          currentUser.username = users[idx].username;
          currentUser.role = users[idx].role;
          persistSession();
          currentUserLabel.textContent = `Přihlášen: ${currentUser.username} (${currentUser.role})`;
          adminControls.style.display = currentUser.role === "admin" ? "flex" : "none";
          renderSidebar();
        }
      }
    } else {
      const id = users.reduce((m, u) => Math.max(m, u.id), 0) + 1;
      users.push({ id, username: usernameVal, password: passwordVal || "", role: roleVal === "admin" ? "admin" : "user" });
    }

    saveUsers();
    renderUsersTable();
    userFormReset();
  });

  // ===== Admin: Manage Collections =====
  btnManageCollections.addEventListener("click", () => {
    if (!requireAdmin()) return;
    collectionFormReset();
    renderCollectionsTable();
    openModal(modalCollections);
  });

  closeCollections.addEventListener("click", () => closeModal(modalCollections));

  function collectionFormReset() {
    collectionId.value = "";
    collectionName.value = "";
    collectionIcon.value = "";
  }

  collectionResetForm.addEventListener("click", collectionFormReset);

  function renderCollectionsTable() {
    collectionsTableBody.innerHTML = "";
    const rows = collections.slice().sort((a, b) => a.name.localeCompare(b.name));
    for (const c of rows) {
      const tr = document.createElement("tr");
      const isActive = c.id === currentCollectionId;
      tr.innerHTML = `
        <td>${c.id}</td>
        <td>${c.icon || "📦"} ${c.name}</td>
        <td>${getTotalCountForCollection(c.id)}</td>
        <td>
          <button class="btn small" data-cedit="${c.id}">Upravit</button>
          <button class="btn small danger" data-cdel="${c.id}" ${isActive ? "disabled" : ""}>Smazat</button>
        </td>
      `;
      collectionsTableBody.appendChild(tr);
    }

    collectionsTableBody.querySelectorAll("[data-cedit]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = String(e.currentTarget.getAttribute("data-cedit"));
        const c = getCollection(id);
        if (!c) return;
        collectionId.value = c.id;
        collectionName.value = c.name;
        collectionIcon.value = c.icon || "";
      });
    });

    collectionsTableBody.querySelectorAll("[data-cdel]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = String(e.currentTarget.getAttribute("data-cdel"));
        if (id === currentCollectionId) return;
        const c = getCollection(id);
        if (!c) return;
        if (!confirm(`Opravdu smazat sbírku "${c.name}"? (Katalog zůstane v localStorage, ale nebude dostupný v UI.)`)) return;
        collections = collections.filter((x) => x.id !== id);
        saveCollections();
        renderCollectionsTable();
        renderSidebar();
      });
    });
  }

  collectionForm.addEventListener("submit", (e) => {
    e.preventDefault();
    if (!requireAdmin()) return;

    const idVal = (collectionId.value || "").trim();
    const nameVal = (collectionName.value || "").trim();
    const iconVal = (collectionIcon.value || "").trim() || "📦";

    if (!nameVal) {
      alert("Název sbírky je povinný.");
      return;
    }

    if (idVal) {
      // Edit
      const idx = collections.findIndex((x) => x.id === idVal);
      if (idx >= 0) {
        collections[idx].name = nameVal;
        collections[idx].icon = iconVal;
      }
    } else {
      // Create
      let base = normalizeCollectionId(nameVal);
      let id = base;
      let i = 2;
      while (collections.some((x) => x.id === id)) {
        id = `${base}-${i}`;
        i++;
      }
      collections.push({ id, name: nameVal, icon: iconVal });

      // Ensure a catalog exists (seed empty if none)
      try {
        const k = catalogKey(id);
        if (!localStorage.getItem(k)) {
          localStorage.setItem(k, JSON.stringify([]));
        }
      } catch (_) {}
    }

    saveCollections();
    renderCollectionsTable();
    renderSidebar();
    collectionFormReset();

    // If current collection was renamed, update titles
    updateAppTitles();
  });

  // ===== Init =====
  ensureDefaultAdmin();
  loadUsers();
  loadCollections();

  // Enable routing (each collection has its own URL)
  window.addEventListener("hashchange", router);

  if (!getCollection(currentCollectionId)) currentCollectionId = collections[0]?.id || "stamps";

  // Auto-login
  if (tryAutoLogin()) {
    if (!getCollection(currentCollectionId)) currentCollectionId = collections[0]?.id || "stamps";
    setAppVisibility(true);
    // Default route if missing
    if (!String(location.hash || "").trim()) {
      setRoute(`collection/${encodeURIComponent(currentCollectionId)}`);
    }
    // Render current collection, then let router correct if URL points elsewhere
    switchCollection(currentCollectionId, { syncRoute: false });
    router();
  } else {
    setAppVisibility(false);
    // Pre-render sidebar data isn't needed when logged out
  }
})();
