(() => {
  const listEl = document.getElementById("list");
  const rtStatus = document.getElementById("rtStatus");
  const shopBoxWrap = document.getElementById("shopCheckboxes");
  const sortEl = document.getElementById("sort");
  const activeOnlyEl = document.getElementById("activeOnly");
  const applyBtn = document.getElementById("apply");
  const clearBtn = document.getElementById("clear");

  const POLL_MS = Number(window.__POLL_MS__ || 3000);

  let allItems = [];
  let current = [];
  let lastLatest = "";
  let lastCount = -1;

  function escapeHtml(str) {
    return String(str ?? "")
      .replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;")
      .replaceAll('"',"&quot;").replaceAll("'","&#039;");
  }
  function escapeAttr(str){ return escapeHtml(str).replaceAll("`","&#096;"); }
  function isActive(endDate){
    const end = new Date(endDate + "T23:59:59");
    return end.getTime() >= Date.now();
  }

  function formatCountdown(ms){
    if (ms <= 0) return "skončeno";
    const totalSec = Math.floor(ms / 1000);
    const days = Math.floor(totalSec / 86400);
    const hrs = Math.floor((totalSec % 86400) / 3600);
    const mins = Math.floor((totalSec % 3600) / 60);
    if (days > 0) return `${days}d ${hrs}h ${mins}m`;
    if (hrs > 0) return `${hrs}h ${mins}m`;
    return `${mins}m`;
  }

  function updateCountdowns(){
    document.querySelectorAll(".countdown[data-end]").forEach(el => {
      const endDate = el.getAttribute("data-end");
      const end = new Date(endDate + "T23:59:59");
      const ms = end.getTime() - Date.now();
      el.textContent = formatCountdown(ms);
    });
  }

  function rowTemplate(c){
    const inactive = !isActive(c.endDate);
    return `
      <div class="item ${inactive ? "inactive" : ""}">
        <div class="itemTop">
          <div class="title">${Number(c.pinned||0) === 1 ? `<span class="pin">📌 Připnuto</span> ` : ``}${escapeHtml(c.productName)}
            ${!inactive ? `<span class="badge ok">aktivní</span>` : `<span class="badge">ukončeno</span>`}
          </div>
          <div class="muted small"><span class="shop-name">${escapeHtml(c.shop)}</span> • končí: <b>${escapeHtml(c.endDate)}</b> • <span class="countdown" data-end="${escapeAttr(c.endDate)}">–</span></div>
        </div>
        <div class="meta">
          <div><b>Počet produktů:</b> ${Number(c.minItems)||0}</div>
          <div><b>Min. útrata:</b> ${Number(c.minSpend)||0} Kč</div>
        </div>
        <div class="prizes">
          <div><b>1. cena:</b> ${escapeHtml(c.prize1)}</div>
          ${c.prize2 ? `<div><b>2. cena:</b> ${escapeHtml(c.prize2)}</div>` : ""}
          ${c.prize3 ? `<div><b>3. cena:</b> ${escapeHtml(c.prize3)}</div>` : ""}
        </div>
        <div class="links">
          <a class="btn small" href="${escapeAttr(c.link)}" target="_blank" rel="noopener">Odkaz na soutěž</a>
        </div>
      </div>
    `;
  }

  function render(){
    if (!current.length) {
      listEl.innerHTML = `<div class="muted">Žádné soutěže pro zvolený filtr.</div>`;
      return;
    }
    listEl.innerHTML = current.map(rowTemplate).join("");
    updateCountdowns();
  }

  function refreshShopCheckboxes(){
    const shops = [...new Set(allItems.map(x => x.shop))].filter(Boolean).sort((a,b)=>a.localeCompare(b,"cs"));
    // zachovej výběr
    const prev = new Set(getSelectedShops());
    shopBoxWrap.innerHTML = shops.map(s => `
      <label class="chk">
        <input type="checkbox" value="${escapeAttr(s)}" ${prev.has(s) ? "checked" : ""}> ${escapeHtml(s)}
      </label>
    `).join("");
  }

  function getSelectedShops(){
    return [...shopBoxWrap.querySelectorAll("input:checked")].map(i => i.value);
  }

  function applyFilters(){
    const selectedShops = getSelectedShops();
    const activeOnly = !!activeOnlyEl?.checked;
    const sort = sortEl.value;

    let items = [...allItems];
    if (selectedShops.length) {
      items = items.filter(x => selectedShops.includes(x.shop));
    }
    if (activeOnly) {
      items = items.filter(x => isActive(x.endDate));
    }

    const sorters = {
      newest: (a,b) => new Date(b.createdAt) - new Date(a.createdAt),
      oldest: (a,b) => new Date(a.createdAt) - new Date(b.createdAt),
      endSoonest: (a,b) => new Date(a.endDate) - new Date(b.endDate),
      endLatest: (a,b) => new Date(b.endDate) - new Date(a.endDate),
    };
    const cmp = sorters[sort] || sorters.newest;

    // pinned vždy nahoře
    items.sort((a,b) => {
      const pa = Number(a.pinned||0), pb = Number(b.pinned||0);
      if (pa !== pb) return pb - pa;
      return cmp(a,b);
    });

    current = items;
    render();
  }

  async function loadAll(){
    // backend také řadí pinned-first; sort posíláme pro konzistenci
    const r = await fetch("./api.php?action=list&sort=" + encodeURIComponent(sortEl.value), { cache:"no-store" });
    allItems = await r.json();
    refreshShopCheckboxes();
    applyFilters();
  }

  async function poll(){
    try{
      const r = await fetch("./api.php?action=latest", { cache:"no-store" });
      const j = await r.json();
      const changed = (j.latest !== lastLatest) || (j.count !== lastCount);
      lastLatest = j.latest; lastCount = j.count;

      rtStatus.textContent = changed ? "Aktualizuji…" : "Sledování změn…";
      if (changed) await loadAll();
      rtStatus.textContent = "Sledování změn…";
    }catch(e){
      rtStatus.textContent = "Problém se spojením…";
    }
  }

  applyBtn.addEventListener("click", applyFilters);
  clearBtn.addEventListener("click", () => {
    [...shopBoxWrap.querySelectorAll("input")].forEach(i => i.checked = false);
    sortEl.value = "newest";
    if (activeOnlyEl) activeOnlyEl.checked = false;
    applyFilters();
  });

  sortEl.addEventListener("change", () => {
    // aby pinned a sort seděl i při příštím pollu, jen přerenderujeme
    applyFilters();
  });

  (async () => {
    await loadAll();
    await poll();
    setInterval(poll, POLL_MS);
    setInterval(updateCountdowns, 30000);
  })();
})(); 
