<?php
require_once __DIR__ . '/lib.php';

// Global JSON error handling (helps on shared hosting where fatal errors would return HTML 500)
@ini_set('display_errors', '0');
@ini_set('log_errors', '1');

function json_fatal_out($msg, $code=500){
  if (headers_sent()) return;
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  echo json_encode(['error'=>'server error','details'=>$msg], JSON_UNESCAPED_UNICODE);
}

set_exception_handler(function($e){
  json_fatal_out($e->getMessage(), 500);
  exit;
});

set_error_handler(function($severity, $message, $file, $line){
  // Convert warnings/notices to exceptions so we get JSON response
  throw new ErrorException($message, 0, $severity, $file, $line);
});

register_shutdown_function(function(){
  $err = error_get_last();
  if (!$err) return;
  $fatal = in_array($err['type'], [E_ERROR,E_PARSE,E_CORE_ERROR,E_COMPILE_ERROR,E_USER_ERROR], true);
  if (!$fatal) return;
  json_fatal_out($err['message'] . ' @' . basename($err['file']) . ':' . $err['line'], 500);
});

$a = $_GET['action'] ?? '';

function is_lock_error($msg): bool {
  $m = strtolower((string)$msg);
  return strpos($m, 'database is locked') !== false || strpos($m, 'busy') !== false || strpos($m, 'locked') !== false;
}


if ($a === 'add') {
  require_token();
  $b = json_decode(file_get_contents('php://input'), true) ?? [];

  $shop = clean($b['shop'] ?? '');
  $productName = clean($b['productName'] ?? '');
  $endDate = clean($b['endDate'] ?? '');
  $minItems = (int)($b['minItems'] ?? 0);
  $minSpend = (int)($b['minSpend'] ?? 0);
  $prize1 = clean($b['prize1'] ?? '');
  $prize2 = clean($b['prize2'] ?? '');
  $prize3 = clean($b['prize3'] ?? '');
  $link = clean($b['link'] ?? '');

  if (!$shop || !$productName || !$endDate || !$prize1 || !$link) json_out(['error'=>'missing fields'],400);
  if (!validate_date($endDate)) json_out(['error'=>'bad endDate'],400);
  if (!validate_url($link)) json_out(['error'=>'bad link'],400);

  $now = now_iso();
  db()->prepare('
    INSERT INTO contests (shop,productName,endDate,minItems,minSpend,prize1,prize2,prize3,link,createdAt,updatedAt,pinned)
    VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
  ')->execute([
    $shop,$productName,$endDate,max(0,$minItems),max(0,$minSpend),
    $prize1,$prize2,$prize3,$link,$now,$now,(int)($b['pinned']??0)
  ]);

  json_out(['ok'=>true]);
}

if ($a === 'update') {
  require_token();
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) json_out(['error'=>'bad id'],400);

  $b = json_decode(file_get_contents('php://input'), true) ?? [];

  $shop = clean($b['shop'] ?? '');
  $productName = clean($b['productName'] ?? '');
  $endDate = clean($b['endDate'] ?? '');
  $minItems = (int)($b['minItems'] ?? 0);
  $minSpend = (int)($b['minSpend'] ?? 0);
  $prize1 = clean($b['prize1'] ?? '');
  $prize2 = clean($b['prize2'] ?? '');
  $prize3 = clean($b['prize3'] ?? '');
  $link = clean($b['link'] ?? '');

  if (!$shop || !$productName || !$endDate || !$prize1 || !$link) json_out(['error'=>'missing fields'],400);
  if (!validate_date($endDate)) json_out(['error'=>'bad endDate'],400);
  if (!validate_url($link)) json_out(['error'=>'bad link'],400);

  $now = now_iso();
  $stmt = db()->prepare('
    UPDATE contests
    SET shop=?, productName=?, endDate=?, minItems=?, minSpend=?, prize1=?, prize2=?, prize3=?, link=?, updatedAt=?, pinned=?
    WHERE id=?
  ');
  $stmt->execute([
    $shop,$productName,$endDate,max(0,$minItems),max(0,$minSpend),
    $prize1,$prize2,$prize3,$link,$now,(int)($b['pinned']??0),$id
  ]);

  json_out(['ok'=>true]);
}

if ($a === 'delete') {
  require_token();
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) json_out(['error'=>'bad id'],400);
  db()->prepare('DELETE FROM contests WHERE id=?')->execute([$id]);
  json_out(['ok'=>true]);
}

if ($a === 'list') {
  $rows = db()->query('SELECT * FROM contests ORDER BY pinned DESC, date(endDate) ASC')->fetchAll();
  json_out($rows);
}


if ($a === 'export') {
  require_token();
  header('Content-Type: application/json; charset=utf-8');
  header('Content-Disposition: attachment; filename="souteze-export.json"');
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');

  $rows = db()->query('SELECT * FROM contests ORDER BY pinned DESC, endDate ASC, datetime(createdAt) DESC')->fetchAll();
  echo json_encode([
    'exportedAt' => now_iso(),
    'version' => 1,
    'contests' => $rows
  ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  exit;
}


if ($a === 'import_file') {
  require_token();
  $mode = $_GET['mode'] ?? 'merge'; // merge | replace
  // allow bigger imports
  @set_time_limit(60);

  if (empty($_FILES['file']) || !is_uploaded_file($_FILES['file']['tmp_name'])) {
    json_out(['error' => 'no file uploaded'], 400);
  }
  $content = file_get_contents($_FILES['file']['tmp_name']);
  $payload = json_decode($content, true);
  if (!$payload) json_out(['error' => 'invalid json'], 400);

  // Reuse the same logic as JSON-body import by injecting into php://input style variable
  $items = $payload['contests'] ?? $payload;
  if (!is_array($items)) json_out(['error' => 'invalid format'], 400);
  if (count($items) > 5000) json_out(['error' => 'too many items'], 400);

  $pdo = db();

  $attempts = 3;
  for ($try = 1; $try <= $attempts; $try++) {
    try {
      // BEGIN IMMEDIATE helps reserve a write lock early
      $pdo->exec('BEGIN IMMEDIATE');

    if ($mode === 'replace') {
      $pdo->exec('DELETE FROM contests');
    }

    $stmtWithId = $pdo->prepare('
      INSERT OR REPLACE INTO contests
      (id, shop, productName, endDate, minItems, minSpend, prize1, prize2, prize3, link, createdAt, updatedAt, pinned, pinnedAt)
      VALUES
      (:id, :shop, :productName, :endDate, :minItems, :minSpend, :prize1, :prize2, :prize3, :link, :createdAt, :updatedAt, :pinned, :pinnedAt)
    ');

    $stmtNoId = $pdo->prepare('
      INSERT INTO contests
      (shop, productName, endDate, minItems, minSpend, prize1, prize2, prize3, link, createdAt, updatedAt, pinned, pinnedAt)
      VALUES
      (:shop, :productName, :endDate, :minItems, :minSpend, :prize1, :prize2, :prize3, :link, :createdAt, :updatedAt, :pinned, :pinnedAt)
    ');

    $now = now_iso();
    $imported = 0;

    foreach ($items as $it) {
      if (!is_array($it)) continue;

      $shop = clean($it['shop'] ?? '');
      $productName = clean($it['productName'] ?? '');
      $endDate = clean($it['endDate'] ?? '');
      $minItems = (int)($it['minItems'] ?? 0);
      $minSpend = (int)($it['minSpend'] ?? 0);
      $prize1 = clean($it['prize1'] ?? '');
      $prize2 = clean($it['prize2'] ?? '');
      $prize3 = clean($it['prize3'] ?? '');
      $link = clean($it['link'] ?? '');
      $pinned = (int)($it['pinned'] ?? 0);
      $pinned = $pinned ? 1 : 0;

      if (!$shop || !$productName || !$endDate || !$prize1 || !$link) continue;
      if (!validate_date($endDate)) continue;
      if (!validate_url($link)) continue;

      $createdAt = clean($it['createdAt'] ?? $now);
      $updatedAt = clean($it['updatedAt'] ?? $now);
      $pinnedAt  = clean($it['pinnedAt'] ?? ($pinned ? $now : ''));
      if (!$pinned) $pinnedAt = '';

      $params = [
        ':shop' => $shop,
        ':productName' => $productName,
        ':endDate' => $endDate,
        ':minItems' => max(0, $minItems),
        ':minSpend' => max(0, $minSpend),
        ':prize1' => $prize1,
        ':prize2' => $prize2,
        ':prize3' => $prize3,
        ':link' => $link,
        ':createdAt' => $createdAt,
        ':updatedAt' => $updatedAt,
        ':pinned' => $pinned,
        ':pinnedAt' => $pinnedAt,
      ];

      $id = (int)($it['id'] ?? 0);
      if ($id > 0) {
        $params[':id'] = $id;
        $stmtWithId->execute($params);
      } else {
        $stmtNoId->execute($params);
      }
      $imported++;
    }

    $pdo->commit();
      json_out(['ok' => true, 'imported' => $imported, 'mode' => $mode]);
    } catch (Throwable $e) {
      if ($pdo->inTransaction()) { $pdo->rollBack(); }
      if ($try < $attempts && is_lock_error($e->getMessage())) {
        usleep(250000 * $try); // 0.25s, 0.5s
        continue;
      }
      json_out(['error' => 'import failed', 'details' => $e->getMessage()], 500);
    }
  }
}


if ($a === 'import') {
  require_token();
  $mode = $_GET['mode'] ?? 'merge'; // merge | replace

  $payload = json_decode(file_get_contents('php://input'), true);
  if (!$payload) json_out(['error' => 'invalid json'], 400);

  $items = $payload['contests'] ?? $payload; // allow raw array or wrapped format
  if (!is_array($items)) json_out(['error' => 'invalid format'], 400);

  if (count($items) > 5000) json_out(['error' => 'too many items'], 400);

  $pdo = db();
  $pdo->beginTransaction();
  try {
    if ($mode === 'replace') {
      $pdo->exec('DELETE FROM contests');
    }

    $stmtWithId = $pdo->prepare('
      INSERT OR REPLACE INTO contests
      (id, shop, productName, endDate, minItems, minSpend, prize1, prize2, prize3, link, createdAt, updatedAt, pinned, pinnedAt)
      VALUES
      (:id, :shop, :productName, :endDate, :minItems, :minSpend, :prize1, :prize2, :prize3, :link, :createdAt, :updatedAt, :pinned, :pinnedAt)
    ');

    $stmtNoId = $pdo->prepare('
      INSERT INTO contests
      (shop, productName, endDate, minItems, minSpend, prize1, prize2, prize3, link, createdAt, updatedAt, pinned, pinnedAt)
      VALUES
      (:shop, :productName, :endDate, :minItems, :minSpend, :prize1, :prize2, :prize3, :link, :createdAt, :updatedAt, :pinned, :pinnedAt)
    ');

    $now = now_iso();
    $imported = 0;

    foreach ($items as $it) {
      if (!is_array($it)) continue;

      $shop = clean($it['shop'] ?? '');
      $productName = clean($it['productName'] ?? '');
      $endDate = clean($it['endDate'] ?? '');
      $minItems = (int)($it['minItems'] ?? 0);
      $minSpend = (int)($it['minSpend'] ?? 0);
      $prize1 = clean($it['prize1'] ?? '');
      $prize2 = clean($it['prize2'] ?? '');
      $prize3 = clean($it['prize3'] ?? '');
      $link = clean($it['link'] ?? '');
      $pinned = (int)($it['pinned'] ?? 0);
      $pinned = $pinned ? 1 : 0;

      if (!$shop || !$productName || !$endDate || !$prize1 || !$link) continue;
      if (!validate_date($endDate)) continue;
      if (!validate_url($link)) continue;

      $createdAt = clean($it['createdAt'] ?? $now);
      $updatedAt = clean($it['updatedAt'] ?? $now);
      $pinnedAt  = clean($it['pinnedAt'] ?? ($pinned ? $now : ''));
      if (!$pinned) $pinnedAt = '';

      $params = [
        ':shop' => $shop,
        ':productName' => $productName,
        ':endDate' => $endDate,
        ':minItems' => max(0, $minItems),
        ':minSpend' => max(0, $minSpend),
        ':prize1' => $prize1,
        ':prize2' => $prize2,
        ':prize3' => $prize3,
        ':link' => $link,
        ':createdAt' => $createdAt,
        ':updatedAt' => $updatedAt,
        ':pinned' => $pinned,
        ':pinnedAt' => $pinnedAt,
      ];

      $id = (int)($it['id'] ?? 0);
      if ($id > 0) {
        $params[':id'] = $id;
        $stmtWithId->execute($params);
      } else {
        $stmtNoId->execute($params);
      }
      $imported++;
    }

    $pdo->commit();
      json_out(['ok' => true, 'imported' => $imported, 'mode' => $mode]);
    } catch (Throwable $e) {
      if ($pdo->inTransaction()) { $pdo->rollBack(); }
      if ($try < $attempts && is_lock_error($e->getMessage())) {
        usleep(250000 * $try); // 0.25s, 0.5s
        continue;
      }
      json_out(['error' => 'import failed', 'details' => $e->getMessage()], 500);
    }
  }
}


if ($a === 'latest') {
  $row = db()->query('SELECT updatedAt FROM contests ORDER BY datetime(updatedAt) DESC LIMIT 1')->fetch();
  $latest = $row ? $row['updatedAt'] : '';
  $count = (int)db()->query('SELECT COUNT(*) AS c FROM contests')->fetch()['c'];
  json_out(['latest'=>$latest,'count'=>$count]);
}

json_out(['error'=>'bad action'],404);
