(() => {
  const listEl = document.getElementById("list");
  const rtStatus = document.getElementById("rtStatus");
  const shopSelect = document.getElementById("shopSelect");
  const sortEl = document.getElementById("sort");
  const applyBtn = document.getElementById("apply");
  const clearBtn = document.getElementById("clear");

  const POLL_MS = Number(window.__POLL_MS__ || 3000);

  let allItems = [];
  let current = [];
  let lastLatest = "";
  let lastCount = -1;

  function escapeHtml(str) {
    return String(str ?? "")
      .replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;")
      .replaceAll('"',"&quot;").replaceAll("'","&#039;");
  }
  function escapeAttr(str){ return escapeHtml(str).replaceAll("`","&#096;"); }
  function isActive(endDate){
    const end = new Date(endDate + "T23:59:59");
    return end.getTime() >= Date.now();
  }

  function rowTemplate(c){
    return `
      <div class="item">
        <div class="itemTop">
          <div class="title">${escapeHtml(c.productName)}
            ${isActive(c.endDate) ? `<span class="badge ok">aktivní</span>` : `<span class="badge">ukončeno</span>`}
          </div>
          <div class="muted small">${escapeHtml(c.shop)} • končí: <b>${escapeHtml(c.endDate)}</b></div>
        </div>
        <div class="meta">
          <div><b>Počet produktů:</b> ${Number(c.minItems)||0}</div>
          <div><b>Min. útrata:</b> ${Number(c.minSpend)||0} Kč</div>
        </div>
        <div class="prizes">
          <div><b>1. cena:</b> ${escapeHtml(c.prize1)}</div>
          ${c.prize2 ? `<div><b>2. cena:</b> ${escapeHtml(c.prize2)}</div>` : ""}
          ${c.prize3 ? `<div><b>3. cena:</b> ${escapeHtml(c.prize3)}</div>` : ""}
        </div>
        <div class="links">
          <a class="btn small" href="${escapeAttr(c.link)}" target="_blank" rel="noopener">Odkaz na soutěž</a>
        </div>
      </div>
    `;
  }

  function render(){
    if (!current.length) {
      listEl.innerHTML = `<div class="muted">Žádné soutěže pro zvolený filtr.</div>`;
      return;
    }
    listEl.innerHTML = current.map(rowTemplate).join("");
  }

  function refreshShopOptions(){
    const shops = [...new Set(allItems.map(x => x.shop))].filter(Boolean).sort((a,b)=>a.localeCompare(b,"cs"));
    const prev = shopSelect.value;
    shopSelect.innerHTML = `<option value="">Všechny</option>` + shops.map(s =>
      `<option value="${escapeAttr(s)}">${escapeHtml(s)}</option>`
    ).join("");
    if (shops.includes(prev)) shopSelect.value = prev;
  }

  function applyFilters(){
    const shop = shopSelect.value;
    const sort = sortEl.value;

    let items = [...allItems];
    if (shop) items = items.filter(x => x.shop === shop);

    const sorters = {
      newest: (a,b) => new Date(b.createdAt) - new Date(a.createdAt),
      oldest: (a,b) => new Date(a.createdAt) - new Date(b.createdAt),
      endSoonest: (a,b) => new Date(a.endDate) - new Date(b.endDate),
      endLatest: (a,b) => new Date(b.endDate) - new Date(a.endDate),
    };
    items.sort(sorters[sort] || sorters.newest);

    current = items;
    render();
  }

  async function loadAll(){
    const r = await fetch("./api.php?action=list", { cache:"no-store" });
    allItems = await r.json();
    refreshShopOptions();
    applyFilters();
  }

  async function poll(){
    try{
      const r = await fetch("./api.php?action=latest", { cache:"no-store" });
      const j = await r.json();
      const changed = (j.latest !== lastLatest) || (j.count !== lastCount);
      lastLatest = j.latest; lastCount = j.count;

      rtStatus.textContent = changed ? "Aktualizuji…" : "Sledování změn…";
      if (changed) await loadAll();
      rtStatus.textContent = "Sledování změn…";
    }catch(e){
      rtStatus.textContent = "Problém se spojením…";
      console.error(e);
    }
  }

  applyBtn.addEventListener("click", applyFilters);
  clearBtn.addEventListener("click", () => {
    shopSelect.value = "";
    sortEl.value = "newest";
    applyFilters();
  });

  (async () => {
    await loadAll();
    await poll();
    setInterval(poll, POLL_MS);
  })();
})(); 
