import { apiGet, apiSend, setStatus } from './ui.js';

let calendar;
let rooms = [];

async function loadRooms(){
  const r = await apiGet('../api/rooms.php');
  rooms = r.rooms;
  const sel = document.getElementById('roomSelect');
  sel.innerHTML = '<option value="0">Všechny pokoje</option>' + rooms.map(x=>`<option value="${x.id}">${x.name}</option>`).join('');
}

function fillModal(fromEvent){
  const form = document.getElementById('editForm');
  const roomSel = form.querySelector('#m_room');
  roomSel.innerHTML = rooms.map(x=>`<option value="${x.id}">${x.name}</option>`).join('');

  if(fromEvent){
    form.querySelector('#m_id').value = fromEvent.id;
    roomSel.value = fromEvent.extendedProps.room_id;
    form.querySelector('#m_type').value = fromEvent.extendedProps.type;
    form.querySelector('#m_start').value = fromEvent.startStr;
    form.querySelector('#m_end').value = fromEvent.endStr;
    form.querySelector('#m_title').value = fromEvent.extendedProps.title || '';
    form.querySelector('#m_name').value = fromEvent.extendedProps.customer_name || '';
    form.querySelector('#m_email').value = fromEvent.extendedProps.email || '';
    form.querySelector('#m_phone').value = fromEvent.extendedProps.phone || '';
    form.querySelector('#m_total').value = fromEvent.extendedProps.total_price || '';
  } else {
    form.reset();
    form.querySelector('#m_id').value = '';
    roomSel.value = String(document.getElementById('roomSelect').value || rooms[0]?.id || 1);
    form.querySelector('#m_type').value = 'reservation';
  }

  toggleFields();
}

function toggleFields(){
  const type = document.getElementById('m_type').value;
  document.getElementById('row_title').style.display = (type === 'block') ? '' : 'none';
  document.getElementById('row_customer').style.display = (type === 'reservation') ? '' : 'none';
}

function showModal(){
  document.getElementById('modal').classList.add('open');
}
function hideModal(){
  document.getElementById('modal').classList.remove('open');
}

async function refreshList(){
  const roomId = Number(document.getElementById('roomSelect').value || 0);
  const now = new Date();
  const start = new Date(now.getFullYear(), now.getMonth(), 1);
  const end = new Date(now.getFullYear(), now.getMonth()+2, 1);
  const fmt = d => d.toISOString().slice(0,10);
  const url = `../api/bookings.php?start=${fmt(start)}&end=${fmt(end)}&room_id=${roomId}`;
  const data = await apiGet(url);
  const rows = data.rows || [];
  const tbody = document.getElementById('listBody');
  tbody.innerHTML = rows.map(r => {
    const who = r.type === 'block' ? (r.title || 'BLOKACE') : (r.customer_name || '');
    return `<tr>
      <td>${r.room_name}</td>
      <td><span class="pill ${r.type}">${r.type}</span></td>
      <td>${r.start_date}</td>
      <td>${r.end_date}</td>
      <td>${who}</td>
      <td>${r.total_price ?? ''}</td>
    </tr>`;
  }).join('') || '<tr><td colspan="6" class="muted">Žádné záznamy</td></tr>';
}

async function initCalendar(){
  const calEl = document.getElementById('calendar');
  calendar = new FullCalendar.Calendar(calEl, {
    initialView: 'dayGridMonth',
    firstDay: 1,
    height: 'auto',
    selectable: true,
    selectMirror: true,
    navLinks: true,
    headerToolbar: { left: 'prev,next today', center: 'title', right: 'dayGridMonth,timeGridWeek' },
    events: async (info, success, failure) => {
      try {
        const roomId = Number(document.getElementById('roomSelect').value || 0);
        const url = `../api/bookings.php?start=${encodeURIComponent(info.startStr)}&end=${encodeURIComponent(info.endStr)}&room_id=${roomId}`;
        const data = await apiGet(url);
        success(data.events);
      } catch (e) { failure(e); }
    },
    select: (sel) => {
      fillModal(null);
      document.getElementById('m_start').value = sel.startStr;
      document.getElementById('m_end').value = sel.endStr;
      showModal();
      setStatus('Vyplň údaje a ulož.');
    },
    eventClick: (info) => {
      fillModal(info.event);
      showModal();
    }
  });
  calendar.render();
}

async function bind(){
  document.getElementById('roomSelect').addEventListener('change', () => {
    calendar.refetchEvents();
    refreshList().catch(console.error);
  });

  document.getElementById('logoutBtn').addEventListener('click', async () => {
    await apiSend('../api/auth.php?action=logout','POST');
    window.location.href = 'login.php';
  });

  document.getElementById('m_type').addEventListener('change', toggleFields);
  document.getElementById('closeModal').addEventListener('click', hideModal);

  document.getElementById('editForm').addEventListener('submit', async (ev) => {
    ev.preventDefault();
    const id = document.getElementById('m_id').value;
    const payload = {
      room_id: Number(document.getElementById('m_room').value),
      type: document.getElementById('m_type').value,
      start_date: document.getElementById('m_start').value,
      end_date: document.getElementById('m_end').value,
      title: document.getElementById('m_title').value.trim(),
      customer_name: document.getElementById('m_name').value.trim(),
      email: document.getElementById('m_email').value.trim(),
      phone: document.getElementById('m_phone').value.trim(),
      total_price: document.getElementById('m_total').value ? Number(document.getElementById('m_total').value) : null,
    };

    try {
      if (id) {
        await apiSend(`../api/bookings.php?id=${encodeURIComponent(id)}`,'PUT',payload);
        setStatus('Uloženo ✅');
      } else {
        await apiSend('../api/bookings.php','POST',payload);
        setStatus('Vytvořeno ✅');
      }
      hideModal();
      calendar.refetchEvents();
      await refreshList();
    } catch (e) {
      setStatus(e.message, true);
    }
  });

  document.getElementById('deleteBtn').addEventListener('click', async () => {
    const id = document.getElementById('m_id').value;
    if (!id) return;
    if (!confirm('Smazat tento záznam?')) return;
    try {
      await apiSend(`../api/bookings.php?id=${encodeURIComponent(id)}`,'DELETE');
      setStatus('Smazáno ✅');
      hideModal();
      calendar.refetchEvents();
      await refreshList();
    } catch (e) {
      setStatus(e.message, true);
    }
  });
}

async function main(){
  await loadRooms();
  await initCalendar();
  await refreshList();
  await bind();
}

main().catch(e => {
  console.error(e);
  setStatus('Chyba při startu dashboardu.', true);
});
