<?php
declare(strict_types=1);

function compute_quote(PDO $pdo, int $roomId, string $start, string $end, string $discountCode = ''): array {
  $stmt = $pdo->prepare('SELECT base_price, currency FROM room_rates WHERE room_id = ?');
  $stmt->execute([$roomId]);
  $rate = $stmt->fetch();
  if (!$rate) {
    return ['ok' => false, 'error' => 'Missing base rate for room'];
  }
  $basePrice = (float)$rate['base_price'];
  $currency = $rate['currency'] ?? 'CZK';

  $seasons = $pdo->query('SELECT name, start_date, end_date, multiplier FROM seasons')->fetchAll();

  $discount = 0.0;
  $discountCode = strtoupper(trim($discountCode));
  if ($discountCode !== '') {
    $d = $pdo->prepare('SELECT percent_off FROM discounts WHERE code = ? AND active = 1 LIMIT 1');
    $d->execute([$discountCode]);
    $row = $d->fetch();
    if ($row) $discount = (float)$row['percent_off'];
  }

  $startDt = new DateTimeImmutable($start);
  $endDt = new DateTimeImmutable($end);
  $subtotal = 0.0;
  $nights = 0;
  for ($d = $startDt; $d < $endDt; $d = $d->modify('+1 day')) {
    $nights++;
    $mult = 1.0;
    foreach ($seasons as $s) {
      if ($d->format('Y-m-d') >= $s['start_date'] && $d->format('Y-m-d') < $s['end_date']) {
        $mult = (float)$s['multiplier'];
        break;
      }
    }
    $subtotal += $basePrice * $mult;
  }

  $discountAmount = round($subtotal * ($discount / 100.0), 2);
  $total = round($subtotal - $discountAmount, 2);

  return [
    'ok' => true,
    'room_id' => $roomId,
    'nights' => $nights,
    'currency' => $currency,
    'subtotal' => round($subtotal, 2),
    'discount_percent' => $discount,
    'discount_amount' => $discountAmount,
    'total' => $total,
  ];
}
