<?php
declare(strict_types=1);
require __DIR__ . '/db.php';
require __DIR__ . '/util.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Public: calc price
if ($method === 'GET' && $action === 'quote') {
  $roomId = (int)($_GET['room_id'] ?? 0);
  $start = (string)($_GET['start_date'] ?? '');
  $end = (string)($_GET['end_date'] ?? '');
  $discountCode = strtoupper(trim((string)($_GET['discount'] ?? '')));

  if ($roomId <= 0) bad_request('room_id required');
  if (!iso_date($start) || !iso_date($end)) bad_request('start_date/end_date required');
  if (strtotime($end) <= strtotime($start)) bad_request('end_date must be after start_date');

  $stmt = $pdo->prepare('SELECT base_price, currency FROM room_rates WHERE room_id = ?');
  $stmt->execute([$roomId]);
  $rate = $stmt->fetch();
  if (!$rate) bad_request('Missing base rate for room');

  $basePrice = (float)$rate['base_price'];
  $currency = $rate['currency'] ?? 'CZK';

  // Load seasons
  $seasons = $pdo->query('SELECT id, name, start_date, end_date, multiplier FROM seasons')->fetchAll();

  // Discount
  $discount = 0.0;
  if ($discountCode !== '') {
    $d = $pdo->prepare('SELECT percent_off FROM discounts WHERE code = ? AND active = 1 LIMIT 1');
    $d->execute([$discountCode]);
    $row = $d->fetch();
    if ($row) $discount = (float)$row['percent_off'];
  }

  $startDt = new DateTimeImmutable($start);
  $endDt = new DateTimeImmutable($end);

  $nights = 0;
  $subtotal = 0.0;
  $breakdown = [];

  for ($d = $startDt; $d < $endDt; $d = $d->modify('+1 day')) {
    $nights++;
    $mult = 1.0;
    $seasonName = null;
    foreach ($seasons as $s) {
      if ($d->format('Y-m-d') >= $s['start_date'] && $d->format('Y-m-d') < $s['end_date']) {
        $mult = (float)$s['multiplier'];
        $seasonName = $s['name'];
        break;
      }
    }
    $nightPrice = $basePrice * $mult;
    $subtotal += $nightPrice;
    $breakdown[] = [
      'date' => $d->format('Y-m-d'),
      'base' => $basePrice,
      'multiplier' => $mult,
      'season' => $seasonName,
      'price' => round($nightPrice, 2),
    ];
  }

  $discountAmount = round($subtotal * ($discount / 100.0), 2);
  $total = round($subtotal - $discountAmount, 2);

  json_out([
    'room_id' => $roomId,
    'nights' => $nights,
    'currency' => $currency,
    'subtotal' => round($subtotal, 2),
    'discount_percent' => $discount,
    'discount_amount' => $discountAmount,
    'total' => $total,
    'breakdown' => $breakdown,
  ]);
}

// Admin: CRUD seasons/discounts/rates
require_admin_only();

if ($method === 'GET' && $action === 'meta') {
  $rates = $pdo->query("SELECT room_id, base_price, currency FROM room_rates ORDER BY room_id")->fetchAll();
  $seasons = $pdo->query("SELECT id, name, start_date, end_date, multiplier FROM seasons ORDER BY start_date")->fetchAll();
  $discounts = $pdo->query("SELECT id, code, name, percent_off, active FROM discounts ORDER BY id DESC")->fetchAll();
  json_out(['rates' => $rates, 'seasons' => $seasons, 'discounts' => $discounts]);
}

if ($method === 'POST' && $action === 'season') {
  $d = read_json_body();
  $name = trim((string)($d['name'] ?? ''));
  $start = (string)($d['start_date'] ?? '');
  $end = (string)($d['end_date'] ?? '');
  $mult = (float)($d['multiplier'] ?? 1.0);
  if ($name === '' || !iso_date($start) || !iso_date($end)) bad_request('name/start_date/end_date required');
  if (strtotime($end) <= strtotime($start)) bad_request('end_date must be after start_date');
  $stmt = $pdo->prepare('INSERT INTO seasons (name, start_date, end_date, multiplier) VALUES (?,?,?,?)');
  $stmt->execute([$name, $start, $end, $mult]);
  json_out(['ok' => true, 'id' => (int)$pdo->lastInsertId()]);
}

if ($method === 'DELETE' && $action === 'season') {
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) bad_request('id required');
  $stmt = $pdo->prepare('DELETE FROM seasons WHERE id = ?');
  $stmt->execute([$id]);
  json_out(['ok' => true]);
}

if ($method === 'POST' && $action === 'discount') {
  $d = read_json_body();
  $code = strtoupper(trim((string)($d['code'] ?? '')));
  $name = trim((string)($d['name'] ?? ''));
  $percent = (float)($d['percent_off'] ?? 0);
  $active = (int)($d['active'] ?? 1);
  if ($code === '' || $name === '') bad_request('code/name required');
  $stmt = $pdo->prepare('INSERT INTO discounts (code, name, percent_off, active) VALUES (?,?,?,?)');
  $stmt->execute([$code, $name, $percent, $active]);
  json_out(['ok' => true, 'id' => (int)$pdo->lastInsertId()]);
}

if ($method === 'DELETE' && $action === 'discount') {
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) bad_request('id required');
  $stmt = $pdo->prepare('DELETE FROM discounts WHERE id = ?');
  $stmt->execute([$id]);
  json_out(['ok' => true]);
}

json_out(['error' => 'Not found'], 404);
