<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

// SQLite "local" database stored as a file inside this project.
// Works well on shared hosting (e.g., Namecheap) because it requires no MySQL setup.
//
// IMPORTANT on hosting:
// - Ensure the /data directory is writable by PHP (CHMOD 755/775/777 depending on host).
// - The database will be auto-created on first request.

// Session for admin API usage
ini_set('session.cookie_httponly', '1');
ini_set('session.use_strict_mode', '1');
ini_set('session.cookie_samesite', 'Lax');

if (PHP_SAPI !== 'cli') {
  session_start();
}

$dataDir = dirname(__DIR__) . '/data';
$dbPath  = $dataDir . '/app.sqlite';

// Ensure data dir exists
if (!is_dir($dataDir)) {
  // @ is intentional: on some hosts permissions may block mkdir; we'll handle later.
  @mkdir($dataDir, 0755, true);
}

try {
  $pdo = new PDO(
    'sqlite:' . $dbPath,
    null,
    null,
    [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]
  );

  // Recommended pragmas
  $pdo->exec('PRAGMA foreign_keys = ON;');
  $pdo->exec('PRAGMA journal_mode = WAL;');

  // Auto-initialize schema if first run
  $stmt = $pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='rooms' LIMIT 1");
  $hasRoomsTable = (bool)$stmt->fetch();

  if (!$hasRoomsTable) {
    $schemaFile = dirname(__DIR__) . '/sql/schema_sqlite.sql';
    if (!file_exists($schemaFile)) {
      throw new RuntimeException('Missing schema_sqlite.sql');
    }
    $sql = file_get_contents($schemaFile);
    if ($sql === false) {
      throw new RuntimeException('Failed to read schema_sqlite.sql');
    }
    $pdo->exec($sql);
  }
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'error' => 'DB connection/init failed',
    'details' => (getenv('APP_DEBUG') ? $e->getMessage() : null)
  ]);
  exit;
}
