<?php
declare(strict_types=1);
require __DIR__ . '/db.php';
require __DIR__ . '/util.php';

$method = $_SERVER['REQUEST_METHOD'];

function overlap_exists(PDO $pdo, int $roomId, string $start, string $end, ?int $ignoreId = null): bool {
  $sql = "SELECT COUNT(*) AS c
          FROM bookings
          WHERE room_id = :room_id
            AND NOT (end_date <= :start_date OR start_date >= :end_date)";
  if ($ignoreId !== null) {
    $sql .= " AND id <> :ignore_id";
  }
  $stmt = $pdo->prepare($sql);
  $params = [
    ':room_id' => $roomId,
    ':start_date' => $start,
    ':end_date' => $end,
  ];
  if ($ignoreId !== null) $params[':ignore_id'] = $ignoreId;
  $stmt->execute($params);
  return ((int)$stmt->fetch()['c']) > 0;
}

// GET is public (for calendar and public availability)
if ($method === 'GET') {
  $start = (string)($_GET['start'] ?? $_GET['start_date'] ?? '');
  $end = (string)($_GET['end'] ?? $_GET['end_date'] ?? '');
  $roomId = isset($_GET['room_id']) ? (int)$_GET['room_id'] : 0;
  $type = (string)($_GET['type'] ?? '');

  if (!iso_date($start) || !iso_date($end)) bad_request('start/end required');

  $sql = "SELECT b.id, b.room_id, b.type, b.start_date, b.end_date, b.title,
                 b.customer_name, b.email, b.phone, b.total_price,
                 r.name AS room_name
          FROM bookings b
          JOIN rooms r ON r.id = b.room_id
          WHERE b.start_date < :end_date AND b.end_date > :start_date";
  $params = [':start_date' => $start, ':end_date' => $end];
  if ($roomId > 0) {
    $sql .= " AND b.room_id = :room_id";
    $params[':room_id'] = $roomId;
  }
  if ($type === 'reservation' || $type === 'block') {
    $sql .= " AND b.type = :type";
    $params[':type'] = $type;
  }
  $sql .= " ORDER BY b.start_date ASC";

  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  // FullCalendar events
  $events = array_map(function($row) {
    $label = $row['type'] === 'block'
      ? ('BLOKACE' . ($row['title'] ? (': ' . $row['title']) : ''))
      : ($row['customer_name'] ?? 'Rezervace');

    return [
      'id' => (int)$row['id'],
      'title' => $row['room_name'] . ' – ' . $label,
      'start' => $row['start_date'],
      'end' => $row['end_date'],
      'allDay' => true,
      'extendedProps' => [
        'room_id' => (int)$row['room_id'],
        'room_name' => $row['room_name'],
        'type' => $row['type'],
        'title' => $row['title'],
        'customer_name' => $row['customer_name'],
        'email' => $row['email'],
        'phone' => $row['phone'],
        'total_price' => $row['total_price'],
      ]
    ];
  }, $rows);

  json_out(['events' => $events, 'rows' => $rows]);
}

// Everything below requires admin login
require_admin();

if ($method === 'POST') {
  $d = read_json_body();
  $roomId = (int)($d['room_id'] ?? 0);
  $type = (string)($d['type'] ?? 'reservation');
  $start = (string)($d['start_date'] ?? '');
  $end = (string)($d['end_date'] ?? '');
  $title = trim((string)($d['title'] ?? ''));
  $name = trim((string)($d['customer_name'] ?? ''));
  $email = trim((string)($d['email'] ?? ''));
  $phone = trim((string)($d['phone'] ?? ''));
  $total = isset($d['total_price']) ? (float)$d['total_price'] : null;

  if ($roomId <= 0) bad_request('room_id required');
  if ($type !== 'reservation' && $type !== 'block') bad_request('type must be reservation|block');
  if (!iso_date($start) || !iso_date($end)) bad_request('start_date/end_date required');
  if (strtotime($end) <= strtotime($start)) bad_request('end_date must be after start_date');

  // reservation needs name; block needs title (optional but recommended)
  if ($type === 'reservation' && $name === '') bad_request('customer_name required for reservation');

  // room exists
  $chk = $pdo->prepare('SELECT id FROM rooms WHERE id = ?');
  $chk->execute([$roomId]);
  if (!$chk->fetch()) bad_request('Room not found');

  if (overlap_exists($pdo, $roomId, $start, $end)) {
    json_out(['error' => 'Room is already booked/blocked in selected dates.'], 409);
  }

  $stmt = $pdo->prepare(
    "INSERT INTO bookings (room_id, type, start_date, end_date, title, customer_name, email, phone, total_price)
     VALUES (?,?,?,?,?,?,?,?,?)"
  );
  $stmt->execute([
    $roomId,
    $type,
    $start,
    $end,
    ($title !== '' ? $title : null),
    ($name !== '' ? $name : null),
    ($email !== '' ? $email : null),
    ($phone !== '' ? $phone : null),
    $total,
  ]);

  json_out(['ok' => true, 'id' => (int)$pdo->lastInsertId()]);
}

if ($method === 'PUT') {
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) bad_request('id required');

  $d = read_json_body();
  $roomId = (int)($d['room_id'] ?? 0);
  $type = (string)($d['type'] ?? 'reservation');
  $start = (string)($d['start_date'] ?? '');
  $end = (string)($d['end_date'] ?? '');
  $title = trim((string)($d['title'] ?? ''));
  $name = trim((string)($d['customer_name'] ?? ''));
  $email = trim((string)($d['email'] ?? ''));
  $phone = trim((string)($d['phone'] ?? ''));
  $total = isset($d['total_price']) ? (float)$d['total_price'] : null;

  if ($roomId <= 0) bad_request('room_id required');
  if ($type !== 'reservation' && $type !== 'block') bad_request('type must be reservation|block');
  if (!iso_date($start) || !iso_date($end)) bad_request('start_date/end_date required');
  if (strtotime($end) <= strtotime($start)) bad_request('end_date must be after start_date');
  if ($type === 'reservation' && $name === '') bad_request('customer_name required for reservation');

  if (overlap_exists($pdo, $roomId, $start, $end, $id)) {
    json_out(['error' => 'Room is already booked/blocked in selected dates.'], 409);
  }

  $stmt = $pdo->prepare(
    "UPDATE bookings
     SET room_id=?, type=?, start_date=?, end_date=?, title=?, customer_name=?, email=?, phone=?, total_price=?
     WHERE id=?"
  );
  $stmt->execute([
    $roomId,
    $type,
    $start,
    $end,
    ($title !== '' ? $title : null),
    ($name !== '' ? $name : null),
    ($email !== '' ? $email : null),
    ($phone !== '' ? $phone : null),
    $total,
    $id,
  ]);

  json_out(['ok' => true]);
}

if ($method === 'DELETE') {
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) bad_request('id required');
  $stmt = $pdo->prepare('DELETE FROM bookings WHERE id = ?');
  $stmt->execute([$id]);
  json_out(['ok' => true]);
}

json_out(['error' => 'Method not allowed'], 405);
