<?php
function cfg() {
  static $cfg = null;
  if ($cfg === null) $cfg = require __DIR__ . '/../config.php';
  return $cfg;
}
function db() : PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $cfg = cfg();
  $dbPath = $cfg['db_path'];
  $dir = dirname($dbPath);
  if (!is_dir($dir)) mkdir($dir, 0775, true);

  $firstRun = !file_exists($dbPath);
  $pdo = new PDO('sqlite:' . $dbPath);
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  if ($firstRun) init_db($pdo);
  return $pdo;
}
function init_db(PDO $pdo) {
  $pdo->exec('PRAGMA journal_mode=WAL;');
  $pdo->exec('
    CREATE TABLE IF NOT EXISTS products(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      barcode TEXT UNIQUE,
      name TEXT NOT NULL,
      photo TEXT,
      default_price REAL DEFAULT 0,
      buy_price REAL DEFAULT 0,
      notes TEXT,
      created_at TEXT DEFAULT CURRENT_TIMESTAMP
    );
  ');

  $pdo->exec('
    CREATE TABLE IF NOT EXISTS product_photos(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      product_id INTEGER NOT NULL,
      path TEXT NOT NULL,
      sort_order INTEGER NOT NULL DEFAULT 0,
      created_at TEXT DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(product_id) REFERENCES products(id)
    );
  ');
  $pdo->exec('CREATE INDEX IF NOT EXISTS idx_photos_product ON product_photos(product_id);');


  $pdo->exec('
    CREATE TABLE IF NOT EXISTS item_attachments(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      item_id INTEGER NOT NULL,
      path TEXT NOT NULL,
      created_at TEXT DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(item_id) REFERENCES box_items(id)
    );
  ');
  $pdo->exec('CREATE INDEX IF NOT EXISTS idx_attach_item ON item_attachments(item_id);');


  // Jednoduchá migrace: pokud existuje products.photo a nejsou záznamy v product_photos, vloží se jako 1. fotka
  try {
    $has = $pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='product_photos'")->fetchColumn();
    if ($has) {
      $cnt = (int)$pdo->query("SELECT COUNT(*) FROM product_photos")->fetchColumn();
      if ($cnt === 0) {
        $rows = $pdo->query("SELECT id, photo FROM products WHERE photo IS NOT NULL AND trim(photo) <> ''")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as $r) {
          $st = $pdo->prepare("INSERT INTO product_photos(product_id, path, sort_order) VALUES(?,?,0)");
          $st->execute([(int)$r['id'], (string)$r['photo']]);
        }
      }
    }
  } catch (Exception $e) { /* ignore */ }

  $pdo->exec('
    CREATE TABLE IF NOT EXISTS boxes(
      id TEXT PRIMARY KEY,
      title TEXT,
      created_at TEXT DEFAULT CURRENT_TIMESTAMP
    );
  ');
  $pdo->exec('
    CREATE TABLE IF NOT EXISTS box_items(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      box_id TEXT NOT NULL,
      product_id INTEGER NOT NULL,
      qty INTEGER NOT NULL DEFAULT 1,
      price REAL NOT NULL DEFAULT 0,
      sold INTEGER NOT NULL DEFAULT 0,
      sold_price REAL,
      sold_at TEXT,
      adv_bazos INTEGER NOT NULL DEFAULT 0,
      adv_sbazar INTEGER NOT NULL DEFAULT 0,
      adv_aukro INTEGER NOT NULL DEFAULT 0,
      adv_other INTEGER NOT NULL DEFAULT 0,
      sold_where TEXT,
      sold_to TEXT,
      sold_note TEXT,
      FOREIGN KEY(box_id) REFERENCES boxes(id),
      FOREIGN KEY(product_id) REFERENCES products(id)
    );
  ');
  $pdo->exec('CREATE INDEX IF NOT EXISTS idx_items_box ON box_items(box_id);');
  $pdo->exec('CREATE INDEX IF NOT EXISTS idx_items_sold ON box_items(sold);');

  
  // Migrace sloupců v products (pro starší DB)
  foreach ([
    "ALTER TABLE products ADD COLUMN buy_price REAL DEFAULT 0"
  ] as $sql) {
    try { $pdo->exec($sql); } catch (Exception $e) { /* ignore */ }
  }


  // Migrace sloupců v box_items (pro starší DB)
  foreach ([
    "ALTER TABLE box_items ADD COLUMN sold_where TEXT",
    "ALTER TABLE box_items ADD COLUMN sold_to TEXT",
    "ALTER TABLE box_items ADD COLUMN sold_note TEXT"
  ] as $sql) {
    try { $pdo->exec($sql); } catch (Exception $e) { /* ignore */ }
  }

}
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function base_url($path=''){
  $cfg = cfg();
  $b = (string)($cfg['base_url'] ?? '');

  // Pokud není base_url nastavené, zkus ho odvodit automaticky z aktuální URL/cesty skriptu.
  if ($b === '' || $b === null) {
    // SCRIPT_NAME např. /inventura11/box.php nebo /box.php
    $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
    $dir = str_replace('\\','/', dirname($script));
    if ($dir === '/' || $dir === '.' ) $dir = '';
    $b = rtrim($dir, '/');

    // Pokud běží přes rewrite a SCRIPT_NAME je nečekané, použij REQUEST_URI (bez query)
    if ($b === '' && !empty($_SERVER['REQUEST_URI'])) {
      $uri = explode('?', (string)$_SERVER['REQUEST_URI'], 2)[0];
      $uri = str_replace('\\','/', $uri);
      // ořízni na složku (bez poslední části)
      $b2 = rtrim(str_replace('\\','/', dirname($uri)), '/');
      if ($b2 !== '/' && $b2 !== '.') $b = $b2;
      else $b = '';
    }
  }

  return rtrim($b,'/') . $path;
}
