<?php
require_once __DIR__ . "/../src/init.php";
ensure_schema();

require_once __DIR__ . "/../src/auth.php";
require_once __DIR__ . "/../src/db.php";
require_once __DIR__ . "/../src/helpers.php";
require_login();

header("Content-Type: application/json; charset=utf-8");

$body = json_decode(file_get_contents("php://input"), true) ?: [];
$action = $_GET["action"] ?? "";

try {
  // --- Orders status ---
  if ($action === "set_status") {
    $id = (int)($body["id"] ?? 0);
    $status = (string)($body["status"] ?? "");
    if ($id<=0 || $status==="") { http_response_code(400); echo json_encode(["ok"=>false]); exit; }

    $st = db()->prepare("UPDATE orders SET status=? WHERE id=?");
    $st->execute([$status, $id]);
    echo json_encode(["ok"=>true]);
    exit;
  }

  // --- Add order item (AJAX) ---
  if ($action === "orderitem_add") {
    $pdo = db();
    $orderId = (int)($body["order_id"] ?? 0);
    $kind = trim((string)($body["kind"] ?? "Materiál"));
    $name = trim((string)($body["name"] ?? ""));
    $unit = trim((string)($body["unit"] ?? "ks"));
    $qty = (float)($body["qty"] ?? 1);
    $price = (int)($body["unit_price_czk"] ?? 0);
    $note = trim((string)($body["note"] ?? ""));

    if ($orderId<=0 || $name==="") { http_response_code(400); echo json_encode(["ok"=>false]); exit; }
    if ($qty<=0) $qty = 1;

    $st = $pdo->prepare("INSERT INTO order_items (order_id,kind,name,unit,qty,unit_price_czk,note) VALUES (?,?,?,?,?,?,?)");
    $st->execute([$orderId,$kind,$name,$unit,$qty,$price,$note]);
    $id = (int)$pdo->lastInsertId();

    $item = $pdo->prepare("SELECT * FROM order_items WHERE id=?");
    $item->execute([$id]);
    $it = $item->fetch();

    $tot = order_totals($orderId);
    echo json_encode(["ok"=>true, "item"=>$it, "totals"=>$tot]);
    exit;
  }

  // --- Delete order item (AJAX) ---
  if ($action === "orderitem_delete") {
    $pdo = db();
    $id = (int)($body["id"] ?? 0);
    $orderId = (int)($body["order_id"] ?? 0);
    if ($id<=0 || $orderId<=0) { http_response_code(400); echo json_encode(["ok"=>false]); exit; }

    $st = $pdo->prepare("DELETE FROM order_items WHERE id=? AND order_id=?");
    $st->execute([$id,$orderId]);

    $tot = order_totals($orderId);
    echo json_encode(["ok"=>true, "totals"=>$tot]);
    exit;
  }

  // --- Pricelist ---
  if ($action === "pricelist_upsert") {
    $pdo = db();
    $id = (int)($body["id"] ?? 0);
    $kind = trim((string)($body["kind"] ?? "Materiál"));
    $name = trim((string)($body["name"] ?? ""));
    $unit = trim((string)($body["unit"] ?? "ks"));
    $price = (int)($body["unit_price_czk"] ?? 0);
    $note = trim((string)($body["note"] ?? ""));
    $isActive = (int)($body["is_active"] ?? 1);

    if ($name === "") { http_response_code(400); echo json_encode(["ok"=>false, "error"=>"missing_name"]); exit; }

    if ($id > 0) {
      $st = $pdo->prepare("UPDATE price_list SET kind=?, name=?, unit=?, unit_price_czk=?, note=?, is_active=? WHERE id=?");
      $st->execute([$kind,$name,$unit,$price,$note,$isActive,$id]);
      echo json_encode(["ok"=>true, "id"=>$id]);
      exit;
    } else {
      $st = $pdo->prepare("INSERT INTO price_list (kind,name,unit,unit_price_czk,note,is_active) VALUES (?,?,?,?,?,?)");
      $st->execute([$kind,$name,$unit,$price,$note,$isActive]);
      echo json_encode(["ok"=>true, "id"=>(int)$pdo->lastInsertId()]);
      exit;
    }
  }

  // --- Travel: calculate distance + route from origin (Drnek 34) ---
  if ($action === "travel_calc") {
    $orderId = (int)($body["order_id"] ?? 0);
    if ($orderId<=0) { http_response_code(400); echo json_encode(["ok"=>false]); exit; }
    require_order_access($orderId);

    $data = update_order_travel($orderId);
    if (!$data) {
      http_response_code(422);
      echo json_encode(["ok"=>false, "error"=>"route_failed"]);
      exit;
    }

    $tot = order_totals($orderId);
    echo json_encode(["ok"=>true, "travel"=>$data, "totals"=>$tot]);
    exit;
  }

  // --- Travel: add (or update) an order item for transport ---
  if ($action === "travel_add_item") {
    $pdo = db();
    $orderId = (int)($body["order_id"] ?? 0);
    if ($orderId<=0) { http_response_code(400); echo json_encode(["ok"=>false]); exit; }
    require_order_access($orderId);

    // ensure travel data exists
    $o = $pdo->prepare("SELECT address, distance_km FROM orders WHERE id=?");
    $o->execute([$orderId]);
    $or = $o->fetch();
    if (!$or) { http_response_code(404); echo json_encode(["ok"=>false]); exit; }
    if ((float)($or["distance_km"] ?? 0) <= 0) {
      $data = update_order_travel($orderId);
      if (!$data) { http_response_code(422); echo json_encode(["ok"=>false, "error"=>"route_failed"]); exit; }
    }

    $cfg = require __DIR__ . "/../src/config.php";
    $pricePerKm = (int)($cfg["travel"]["price_per_km_czk"] ?? 0);

    $o2 = $pdo->prepare("SELECT distance_km, address FROM orders WHERE id=?");
    $o2->execute([$orderId]);
    $or2 = $o2->fetch();
    $km = (float)($or2["distance_km"] ?? 0);
    $name = "Doprava (" . ($cfg["travel"]["origin_address"] ?? "") . " → " . ($or2["address"] ?? "") . ")";

    // upsert by exact kind+unit and name prefix to avoid duplicates
    $find = $pdo->prepare("SELECT id FROM order_items WHERE order_id=? AND kind='Doprava' AND unit='km' AND name LIKE 'Doprava (%' LIMIT 1");
    $find->execute([$orderId]);
    $existing = $find->fetch();

    if ($existing) {
      $upd = $pdo->prepare("UPDATE order_items SET name=?, qty=?, unit_price_czk=? WHERE id=?");
      $upd->execute([$name, $km, $pricePerKm, (int)$existing["id"]]);
    } else {
      $ins = $pdo->prepare("INSERT INTO order_items (order_id,kind,name,unit,qty,unit_price_czk,note) VALUES (?,?,?,?,?,?,?)");
      $ins->execute([$orderId, "Doprava", $name, "km", $km, $pricePerKm, "vypočteno dle vzdálenosti"]);
    }

    $tot = order_totals($orderId);
    echo json_encode(["ok"=>true, "totals"=>$tot]);
    exit;
  }

  if ($action === "pricelist_delete") {
    $pdo = db();
    $id = (int)($body["id"] ?? 0);
    if ($id<=0) { http_response_code(400); echo json_encode(["ok"=>false]); exit; }
    $st = $pdo->prepare("DELETE FROM price_list WHERE id=?");
    $st->execute([$id]);
    echo json_encode(["ok"=>true]);
    exit;
  }

  http_response_code(404);
  echo json_encode(["ok"=>false, "error"=>"Unknown action"]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(["ok"=>false, "error"=>"Server error"]);
}
