<?php
declare(strict_types=1);

require_once __DIR__ . "/db.php";

function ensure_schema(): void {
  $pdo = db();

  $check = $pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='users'")->fetch();
  if ($check) {
    // Lightweight migrations for existing installs (add new columns when missing)
    $cols = $pdo->query("PRAGMA table_info(orders)")->fetchAll();
    $have = [];
    foreach ($cols as $c) { $have[$c['name']] = true; }

    if (!isset($have['dest_lat']))         $pdo->exec("ALTER TABLE orders ADD COLUMN dest_lat REAL");
    if (!isset($have['dest_lng']))         $pdo->exec("ALTER TABLE orders ADD COLUMN dest_lng REAL");
    if (!isset($have['distance_km']))      $pdo->exec("ALTER TABLE orders ADD COLUMN distance_km REAL");
    if (!isset($have['travel_cost_czk']))  $pdo->exec("ALTER TABLE orders ADD COLUMN travel_cost_czk INTEGER");
    if (!isset($have['travel_roundtrip'])) $pdo->exec("ALTER TABLE orders ADD COLUMN travel_roundtrip INTEGER NOT NULL DEFAULT 0");
    if (!isset($have['route_geojson']))    $pdo->exec("ALTER TABLE orders ADD COLUMN route_geojson TEXT");
    if (!isset($have['route_updated_at'])) $pdo->exec("ALTER TABLE orders ADD COLUMN route_updated_at TEXT");
    return;
  }

  $pdo->exec("
    CREATE TABLE users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      email TEXT NOT NULL UNIQUE,
      password_hash TEXT NOT NULL,
      role TEXT NOT NULL CHECK(role IN ('admin','office','montér')) DEFAULT 'montér',
      is_active INTEGER NOT NULL DEFAULT 1,
      created_at TEXT DEFAULT (datetime('now'))
    );

    CREATE TABLE clients (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      phone TEXT,
      email TEXT,
      address TEXT,
      ico TEXT,
      dic TEXT,
      note TEXT,
      created_at TEXT DEFAULT (datetime('now'))
    );

    CREATE TABLE orders (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      client_id INTEGER NOT NULL,
      title TEXT NOT NULL,
      address TEXT NOT NULL,
      dest_lat REAL,
      dest_lng REAL,
      distance_km REAL,
      travel_cost_czk INTEGER,
      travel_roundtrip INTEGER NOT NULL DEFAULT 0,
      route_geojson TEXT,
      route_updated_at TEXT,
      flooring_type TEXT NOT NULL,
      area_m2 REAL NOT NULL DEFAULT 0,
      status TEXT NOT NULL CHECK(status IN ('Čeká','V běhu','Hotovo','Zrušeno')) DEFAULT 'Čeká',
      start_date TEXT,
      end_date TEXT,
      assigned_user_id INTEGER,
      note TEXT,
      vat_rate REAL NOT NULL DEFAULT 21.0,
      discount_czk INTEGER NOT NULL DEFAULT 0,

      created_at TEXT DEFAULT (datetime('now')),
      updated_at TEXT DEFAULT (datetime('now')),
      FOREIGN KEY(client_id) REFERENCES clients(id) ON DELETE RESTRICT,
      FOREIGN KEY(assigned_user_id) REFERENCES users(id) ON DELETE SET NULL
    );

    CREATE TABLE order_items (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      order_id INTEGER NOT NULL,
      kind TEXT NOT NULL CHECK(kind IN ('Materiál','Práce','Doprava','Ostatní')) DEFAULT 'Materiál',
      name TEXT NOT NULL,
      unit TEXT NOT NULL DEFAULT 'ks',
      qty REAL NOT NULL DEFAULT 1.0,
      unit_price_czk INTEGER NOT NULL DEFAULT 0,
      note TEXT,
      FOREIGN KEY(order_id) REFERENCES orders(id) ON DELETE CASCADE
    );

    CREATE TABLE price_list (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      kind TEXT NOT NULL CHECK(kind IN ('Materiál','Práce','Doprava','Ostatní')) DEFAULT 'Materiál',
      name TEXT NOT NULL,
      unit TEXT NOT NULL DEFAULT 'ks',
      unit_price_czk INTEGER NOT NULL DEFAULT 0,
      is_active INTEGER NOT NULL DEFAULT 1,
      note TEXT,
      created_at TEXT DEFAULT (datetime('now'))
    );
  ");

  // default admin
  $email = "admin@local";
  $pass = "admin123";
  $hash = password_hash($pass, PASSWORD_DEFAULT);

  $stmt = $pdo->prepare("INSERT INTO users (name,email,password_hash,role,is_active) VALUES (?,?,?,?,1)");
  $stmt->execute(["Admin", $email, $hash, "admin"]);

  // default office účet
  $email2 = "office@local";
  $pass2 = "office123";
  $hash2 = password_hash($pass2, PASSWORD_DEFAULT);
  $stmt->execute(["Office", $email2, $hash2, "office"]);

  // seed ceník (ukázkové položky)
  $seed = $pdo->prepare("INSERT INTO price_list (kind,name,unit,unit_price_czk,note,is_active) VALUES (?,?,?,?,?,1)");
  $seed->execute(["Práce","Pokládka vinylu","m²",450,"včetně přípravy podkladu"]);
  $seed->execute(["Práce","Pokládka laminátu","m²",380,""]);
  $seed->execute(["Práce","Demontáž staré podlahy","m²",120,""]);
  $seed->execute(["Materiál","Nivelace (směs)","kg",35,"orientační"]);
  $seed->execute(["Doprava","Doprava po městě","km",18,""]);
  $seed->execute(["Ostatní","Soklové lišty – montáž","bm",55,""]);

  // demo klient a zakázka
  $pdo->prepare("INSERT INTO clients (name,phone,address) VALUES (?,?,?)")
      ->execute(["Jan Dvořák","+420 606 111 222","Praha 6, Bělohorská 12"]);
  $clientId = (int)$pdo->lastInsertId();

  $pdo->prepare("INSERT INTO orders (client_id,title,address,flooring_type,area_m2,status,start_date,end_date,note) VALUES (?,?,?,?,?,?,?,?,?)")
      ->execute([$clientId,"Pokládka vinylu", "Praha 6, Bělohorská 12","Vinyl",42,"V běhu","2026-01-08","2026-01-10","Demontáž staré podlahy + lišty."]);

  $orderId = (int)$pdo->lastInsertId();

  $pdo->prepare("INSERT INTO order_items (order_id,kind,name,unit,qty,unit_price_czk,note) VALUES (?,?,?,?,?,?,?)")
      ->execute([$orderId,"Práce","Pokládka vinylu","m²",42,450,"včetně přípravy podkladu"]);
  $pdo->prepare("INSERT INTO order_items (order_id,kind,name,unit,qty,unit_price_czk,note) VALUES (?,?,?,?,?,?,?)")
      ->execute([$orderId,"Materiál","Vinyl (dodávka)","m²",42,520,"dle výběru zákazníka"]);
}
