<?php
declare(strict_types=1);

require_once __DIR__ . "/db.php";

function h(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, "UTF-8");
}

function czk(int|float $n): string {
  $v = (float)$n;
  return number_format($v, 0, ",", " ") . " Kč";
}

function order_totals(int $orderId): array {
  $pdo = db();

  $stmt = $pdo->prepare("SELECT COALESCE(SUM(qty * unit_price_czk),0) AS subtotal FROM order_items WHERE order_id=?");
  $stmt->execute([$orderId]);
  $subtotal = (float)$stmt->fetch()["subtotal"];

  $o = $pdo->prepare("SELECT vat_rate, discount_czk FROM orders WHERE id=?");
  $o->execute([$orderId]);
  $row = $o->fetch();
  $vatRate = (float)($row["vat_rate"] ?? 21.0);
  $discount = (int)($row["discount_czk"] ?? 0);

  $afterDiscount = max(0.0, $subtotal - $discount);
  $vat = $afterDiscount * ($vatRate / 100.0);
  $total = $afterDiscount + $vat;

  return [
    "subtotal" => $subtotal,
    "discount" => $discount,
    "after_discount" => $afterDiscount,
    "vat_rate" => $vatRate,
    "vat" => $vat,
    "total" => $total,
  ];
}

/**
 * Jednoduchý HTTP GET, vrací decoded JSON (pole) nebo null.
 * Pozn.: používá veřejné služby OpenStreetMap (Nominatim) a OSRM demo server.
 */
function http_get_json(string $url, int $timeoutSec = 10): ?array {
  $cfg = require __DIR__ . "/config.php";
  $ua = "PodlaharstviApp/1.0 (" . ($cfg["app"]["company_email"] ?? "") . ")";
  $ctx = stream_context_create([
    "http" => [
      "method" => "GET",
      "header" => "User-Agent: {$ua}\r\nAccept: application/json\r\n",
      "timeout" => $timeoutSec,
    ]
  ]);

  $raw = @file_get_contents($url, false, $ctx);
  if ($raw === false) return null;
  $j = json_decode($raw, true);
  return is_array($j) ? $j : null;
}

/** Geocode adresa přes Nominatim. Vrací [lat,lng] nebo null. */
function geocode_address(string $address): ?array {
  $q = trim($address);
  if ($q === "") return null;
  $url = "https://nominatim.openstreetmap.org/search?format=jsonv2&limit=1&countrycodes=cz&q=" . rawurlencode($q);
  $res = http_get_json($url, 12);
  if (!$res || !isset($res[0]["lat"], $res[0]["lon"])) return null;
  return [
    "lat" => (float)$res[0]["lat"],
    "lng" => (float)$res[0]["lon"],
    "display_name" => (string)($res[0]["display_name"] ?? "")
  ];
}

/**
 * Routing přes OSRM (driving). Vrací [distance_km, geometry_geojson] nebo null.
 */
function route_osrm(float $fromLat, float $fromLng, float $toLat, float $toLng): ?array {
  $url = "https://router.project-osrm.org/route/v1/driving/".
         rawurlencode($fromLng.",".$fromLat.";".$toLng.",".$toLat) .
         "?overview=full&geometries=geojson";
  $res = http_get_json($url, 15);
  if (!$res || ($res["code"] ?? "") !== "Ok" || empty($res["routes"][0])) return null;
  $r = $res["routes"][0];
  $meters = (float)($r["distance"] ?? 0);
  $km = $meters / 1000.0;
  return [
    "distance_km" => $km,
    "geometry" => $r["geometry"] ?? null,
    "duration_sec" => (float)($r["duration"] ?? 0),
  ];
}

/**
 * Spočítá vzdálenost + trasu pro zakázku, uloží do orders a vrátí data.
 */
function update_order_travel(int $orderId, ?bool $roundtripOverride = null): ?array {
  $pdo = db();
  $cfg = require __DIR__ . "/config.php";

  $o = $pdo->prepare("SELECT id, address FROM orders WHERE id=?");
  $o->execute([$orderId]);
  $order = $o->fetch();
  if (!$order) return null;

  $originAddr = (string)($cfg["travel"]["origin_address"] ?? "");
  $origin = geocode_address($originAddr);
  $dest = geocode_address((string)$order["address"]);
  if (!$origin || !$dest) return null;

  $route = route_osrm($origin["lat"], $origin["lng"], $dest["lat"], $dest["lng"]);
  if (!$route || empty($route["geometry"])) return null;

  $km = (float)$route["distance_km"];
  $roundtrip = $roundtripOverride !== null
    ? (bool)$roundtripOverride
    : (bool)($cfg["travel"]["roundtrip"] ?? false);
  if ($roundtrip) $km = $km * 2.0;

  $pricePerKm = (int)($cfg["travel"]["price_per_km_czk"] ?? 0);
  $cost = (int)round($km * $pricePerKm);

  $geojson = json_encode([
    "type" => "Feature",
    "properties" => [
      "origin" => $originAddr,
      "destination" => (string)$order["address"],
      "origin_lat" => $origin["lat"],
      "origin_lng" => $origin["lng"],
      "dest_lat" => $dest["lat"],
      "dest_lng" => $dest["lng"],
    ],
    "geometry" => $route["geometry"],
  ], JSON_UNESCAPED_UNICODE);

  $st = $pdo->prepare("UPDATE orders SET dest_lat=?, dest_lng=?, distance_km=?, travel_cost_czk=?, travel_roundtrip=?, route_geojson=?, route_updated_at=datetime('now'), updated_at=datetime('now') WHERE id=?");
  $st->execute([$dest["lat"], $dest["lng"], $km, $cost, $roundtrip ? 1 : 0, $geojson, $orderId]);

  $t = $pdo->prepare("SELECT route_updated_at FROM orders WHERE id=?");
  $t->execute([$orderId]);
  $ru = (string)($t->fetch()["route_updated_at"] ?? "");

  return [
    "origin" => $origin,
    "destination" => $dest,
    "distance_km" => $km,
    "price_per_km_czk" => $pricePerKm,
    "travel_cost_czk" => $cost,
    "route_geojson" => $geojson,
    "route_updated_at" => $ru,
    "roundtrip" => $roundtrip,
  ];
}
