/* Simple WebXR VR wall gallery (Three.js) */
let renderer, scene, camera, controller, raycaster, tempMatrix;
let panels=[], items=[];
let screenMesh, screenMat, screenTex, videoEl;

const msg=document.getElementById('msg');
const btn=document.getElementById('enterVR');

function show(html){msg.style.display='';msg.innerHTML=html;}

async function fetchItems(){
  const res=await fetch('api/media.php',{cache:'no-store'});
  const data=await res.json();
  items=(data.items||[]).slice(0,80);
}

function tex(url){
  return new Promise((resolve,reject)=>{
    new THREE.TextureLoader().load(url,t=>{t.colorSpace=THREE.SRGBColorSpace;resolve(t);},undefined,reject);
  });
}

function setup(){
  scene=new THREE.Scene();
  scene.background=new THREE.Color(0x050812);
  camera=new THREE.PerspectiveCamera(70, innerWidth/innerHeight, 0.05, 100);
  camera.position.set(0,1.6,2.2);

  scene.add(new THREE.HemisphereLight(0xffffff,0x223344,0.9));
  const d=new THREE.DirectionalLight(0xffffff,0.7); d.position.set(2,5,2); scene.add(d);

  const floor=new THREE.Mesh(new THREE.PlaneGeometry(20,20), new THREE.MeshStandardMaterial({color:0x0b1020, roughness:1}));
  floor.rotation.x=-Math.PI/2; scene.add(floor);

  screenMat=new THREE.MeshBasicMaterial({color:0x111111});
  screenMesh=new THREE.Mesh(new THREE.PlaneGeometry(2.2,1.3), screenMat);
  screenMesh.position.set(0,1.6,-2.2);
  scene.add(screenMesh);

  raycaster=new THREE.Raycaster();
  tempMatrix=new THREE.Matrix4();

  renderer=new THREE.WebGLRenderer({antialias:true});
  renderer.setPixelRatio(devicePixelRatio);
  renderer.setSize(innerWidth, innerHeight);
  renderer.xr.enabled=true;
  document.body.appendChild(renderer.domElement);

  controller=renderer.xr.getController(0);
  controller.addEventListener('selectstart', onSelect);
  scene.add(controller);

  const line=new THREE.Line(
    new THREE.BufferGeometry().setFromPoints([new THREE.Vector3(0,0,0),new THREE.Vector3(0,0,-1)]),
    new THREE.LineBasicMaterial({color:0xffffff})
  );
  line.scale.z=2; controller.add(line);

  addEventListener('resize', ()=>{camera.aspect=innerWidth/innerHeight;camera.updateProjectionMatrix();renderer.setSize(innerWidth,innerHeight);});
}

async function build(){
  const radius=2.6, rows=3, cols=9;
  const w=0.32, h=0.20, gap=0.06;
  const a0=-0.75, a1=0.75;
  const total=Math.min(items.length, rows*cols);

  for(let i=0;i<total;i++){
    const it=items[i];
    const col=i%cols, row=Math.floor(i/cols);
    const t=col/(cols-1);
    const ang=a0+(a1-a0)*t;
    const x=Math.sin(ang)*radius;
    const z=-Math.cos(ang)*radius;
    const y=1.15+(rows-1-row)*(h+gap);

    const group=new THREE.Group();
    group.position.set(x,y,z);
    group.lookAt(0,y,0);

    const frame=new THREE.Mesh(new THREE.PlaneGeometry(w+0.03,h+0.03), new THREE.MeshBasicMaterial({color:0x0f1830}));
    group.add(frame);

    const mat=new THREE.MeshBasicMaterial({color:0x222222});
    const panel=new THREE.Mesh(new THREE.PlaneGeometry(w,h), mat);
    panel.position.z=0.001;
    panel.userData.item=it;
    group.add(panel);
    panels.push(panel);
    scene.add(group);

    try{
      const ttex=await tex(it.thumb||it.url);
      mat.map=ttex; mat.needsUpdate=true;
    }catch{}
  }

  if(items[0]) await open(items[0]);
}

async function open(it){
  if(videoEl){ try{videoEl.pause();}catch{} videoEl.remove(); videoEl=null; }
  if(screenTex){ try{screenTex.dispose();}catch{} screenTex=null; }

  if(it.type==='video'){
    const v=document.createElement('video');
    v.src=it.url; v.loop=true; v.muted=true; v.playsInline=true; v.preload='auto';
    v.style.position='fixed'; v.style.left='-9999px';
    document.body.appendChild(v);
    await v.play().catch(()=>{});
    videoEl=v;
    screenTex=new THREE.VideoTexture(v);
    screenTex.colorSpace=THREE.SRGBColorSpace;
    screenMat.map=screenTex; screenMat.color.set(0xffffff); screenMat.needsUpdate=true;
  }else{
    const ttex=await tex(it.url);
    screenTex=ttex;
    screenMat.map=ttex; screenMat.color.set(0xffffff); screenMat.needsUpdate=true;
  }
}

function onSelect(){
  tempMatrix.identity().extractRotation(controller.matrixWorld);
  raycaster.ray.origin.setFromMatrixPosition(controller.matrixWorld);
  raycaster.ray.direction.set(0,0,-1).applyMatrix4(tempMatrix);
  const hits=raycaster.intersectObjects(panels,false);
  if(hits.length){
    const it=hits[0].object.userData.item;
    if(it) open(it);
  }
}

function animate(){ renderer.setAnimationLoop(()=>renderer.render(scene,camera)); }

async function main(){
  if(!navigator.xr){ show('<b>WebXR není dostupné.</b><br/>Otevři v Meta Quest Browseru.'); btn.disabled=true; return; }
  await fetchItems();
  setup();
  await build();
  animate();

  btn.addEventListener('click', async ()=>{
    try{
      const ok=await navigator.xr.isSessionSupported('immersive-vr');
      if(!ok){ show('<b>Immersive VR není podporované.</b>'); return; }
      const session=await navigator.xr.requestSession('immersive-vr',{optionalFeatures:['local-floor','bounded-floor','hand-tracking']});
      renderer.xr.setSession(session);
    }catch(e){ show('<b>Nepodařilo se spustit VR.</b><br/>'+(e?.message||e)); }
  });
}
main();
