const API_URL = "api.php";

/**
 * items:
 * {id, period, date, desc, amount(+/-), type, kind(txn|subtotal|note)}
 *
 * cash:
 * { denoms: { "1": n, ... }, total }
 *
 * loans:
 * { id, name, principal, monthly, firstDate(dd.mm.yyyy), payments:[{date,amount}] }
 */
let items = [];
let cash = { denoms: {}, total: 0 };
let loans = [];

const el = (id) => document.getElementById(id);

function czk(n) {
  const v = Number(n || 0);
  return v.toLocaleString("cs-CZ", { style: "currency", currency: "CZK" });
}

function uid() {
  return Math.random().toString(16).slice(2) + Date.now().toString(16);
}

function escapeHtml(s){
  return String(s ?? "")
    .replaceAll("&","&amp;")
    .replaceAll("<","&lt;")
    .replaceAll(">","&gt;")
    .replaceAll('"',"&quot;")
    .replaceAll("'","&#039;");
}

// ---------- parsing amounts / periods ----------
function normalizeAmountFromCz(text) {
  // "1 217,00 Kč" -> 1217.00
  if (!text) return null;
  const s = text
    .replace(/\u00A0/g, " ")
    .replace(/\s/g, "")
    .replace("Kč", "")
    .replace(",", ".")
    .trim();
  const num = Number(s);
  return Number.isFinite(num) ? num : null;
}

function extractAllAmounts(line) {
  const out = [];
  const re = /-?\d[\d\s\u00A0]*,\d{2}\s*Kč/g;
  let m;
  while ((m = re.exec(line)) !== null) {
    const raw = m[0];
    const value = normalizeAmountFromCz(raw);
    if (value !== null) out.push({ raw, value, index: m.index });
  }
  return out;
}

function extractDate(line) {
  const m = line.match(/\b(\d{1,2}\.\d{1,2}\.\d{4})\b/);
  return m ? m[1] : "";
}

function splitColumns(line) {
  if (line.includes("\t")) {
    return line.split("\t").map(s => s.trim()).filter(s => s !== "");
  }
  return line.split(/\s{2,}/).map(s => s.trim()).filter(s => s !== "");
}

function isSubtotalLine(line) {
  const s = line.toLowerCase();
  return s.includes("via") || s.includes("bez pp") || s.includes("plus") || s.includes("saldo");
}

function parseRaw(raw) {
  const lines = raw
    .split(/\r?\n/)
    .map(l => l.replace(/\u00A0/g, " ").trimEnd())
    .filter(l => l.trim() !== "");

  let currentPeriod = "";
  const out = [];

  for (const originalLine of lines) {
    const line = originalLine.trim();

    const mPeriod = line.match(/^(konec|začátek)\s+(.+?)(\s+(-?\d[\d\s]*,\d{2})\s*Kč)?$/i);
    if (mPeriod) {
      currentPeriod = `${mPeriod[1]} ${mPeriod[2]}`.trim();
      if (mPeriod[4]) {
        const amt = normalizeAmountFromCz(mPeriod[4] + " Kč");
        if (amt !== null) {
          out.push({
            id: uid(),
            period: currentPeriod,
            date: "",
            desc: `Saldo: ${currentPeriod}`,
            amount: amt,
            type: amt >= 0 ? "income" : "expense",
            kind: "subtotal"
          });
        }
      }
      continue;
    }

    const cols = splitColumns(line);
    const date = extractDate(line);
    const amounts = extractAllAmounts(line);

    // Poznámky bez "Kč" ignorujeme (jak chceš)
    if (amounts.length === 0) continue;

    if (isSubtotalLine(line)) {
      const last = amounts[amounts.length - 1];
      let desc = line.replace(last.raw, "").replace(/\s+/g, " ").trim();
      if (date) desc = desc.replace(date, "").trim();
      out.push({
        id: uid(),
        period: currentPeriod,
        date,
        desc: desc || "Mezistav / součet",
        amount: last.value,
        type: last.value >= 0 ? "income" : "expense",
        kind: "subtotal"
      });
      continue;
    }

    // výdaje s datem -> ukládáme jako záporné
    if (date) {
      const last = amounts[amounts.length - 1];

      let desc = line.replace(last.raw, "").replace(/\s+/g, " ").trim();
      desc = desc.replace(date, "").replace(/\s+/g, " ").trim();

      if (!desc || desc.length < 2) {
        desc = cols
          .filter(c => !c.includes(date) && !c.match(/-?\d[\d\s]*,\d{2}\s*Kč/))
          .join(" ")
          .trim();
        if (!desc) desc = "Výdaj";
      }

      out.push({
        id: uid(),
        period: currentPeriod,
        date,
        desc,
        amount: -Math.abs(last.value),
        type: "expense",
        kind: "txn"
      });

      // případný zůstatek z řádku jako subtotal
      if (amounts.length >= 2) {
        const maybeBalance = amounts[amounts.length - 2];
        if (maybeBalance.index < last.index) {
          out.push({
            id: uid(),
            period: currentPeriod,
            date,
            desc: "Mezistav / zůstatek (z řádku)",
            amount: maybeBalance.value,
            type: maybeBalance.value >= 0 ? "income" : "expense",
            kind: "subtotal"
          });
        }
      }
      continue;
    }

    // příjmy bez data
    const main = amounts[0];
    let desc = line.replace(main.raw, "").replace(/\s+/g, " ").trim();
    if (!desc) desc = "Příjem";

    out.push({
      id: uid(),
      period: currentPeriod,
      date: "",
      desc,
      amount: Math.abs(main.value),
      type: "income",
      kind: "txn"
    });

    // další částky v řádku: txn; poslední velká jako subtotal (heuristika)
    if (amounts.length >= 2) {
      for (let i = 1; i < amounts.length; i++) {
        const a = amounts[i];
        const isLast = i === amounts.length - 1;
        if (isLast && a.value >= 20000) {
          out.push({
            id: uid(),
            period: currentPeriod,
            date: "",
            desc: "Součet / mezistav (z řádku)",
            amount: a.value,
            type: a.value >= 0 ? "income" : "expense",
            kind: "subtotal"
          });
        } else {
          out.push({
            id: uid(),
            period: currentPeriod,
            date: "",
            desc: "(další částka z řádku)",
            amount: Math.abs(a.value),
            type: "income",
            kind: "txn"
          });
        }
      }
    }
  }

  return out;
}

// ---------- summary / filtering ----------
function computeSummary(list) {
  let income = 0, expense = 0;
  for (const it of list) {
    if ((it.kind || "txn") !== "txn") continue;
    if (it.amount >= 0) income += it.amount;
    else expense += Math.abs(it.amount);
  }
  return { income, expense, net: income - expense };
}

function getFiltered() {
  const q = el("filterText").value.trim().toLowerCase();
  const t = el("filterType").value;

  return items.filter(it => {
    const matchText =
      !q ||
      (it.desc || "").toLowerCase().includes(q) ||
      (it.period || "").toLowerCase().includes(q) ||
      (it.kind || "").toLowerCase().includes(q);

    const matchType =
      t === "all" ||
      (t === "income" && it.amount >= 0) ||
      (t === "expense" && it.amount < 0);

    return matchText && matchType;
  });
}

// ---------- cash ----------
function getCashInputs() {
  return Array.from(document.querySelectorAll(".cashCount"));
}

function recalcCash() {
  const inputs = getCashInputs();
  let total = 0;
  const denoms = {};
  for (const inp of inputs) {
    const denom = Number(inp.getAttribute("data-denom"));
    const countRaw = (inp.value || "").trim();
    const count = countRaw === "" ? 0 : Number(countRaw);
    const safeCount = Number.isFinite(count) ? Math.max(0, Math.floor(count)) : 0;

    denoms[String(denom)] = safeCount;
    const sub = denom * safeCount;
    total += sub;

    const subEl = document.querySelector(`.cashSub[data-sub="${denom}"]`);
    if (subEl) subEl.textContent = czk(sub);
  }

  cash = { denoms, total };

  const tEl = document.getElementById("cashTotal");
  if (tEl) tEl.textContent = czk(total);

  const sEl = document.getElementById("sumCash");
  if (sEl) sEl.textContent = czk(total);
}

function fillCashFromData(c) {
  const inputs = getCashInputs();
  const denoms = (c && c.denoms && typeof c.denoms === "object") ? c.denoms : {};
  for (const inp of inputs) {
    const denom = inp.getAttribute("data-denom");
    inp.value = String(denoms[denom] ?? "");
  }
  recalcCash();
}

// ---------- loans ----------
function parseCzNumber(v) {
  const s = String(v ?? "").trim().replace(/\s+/g, "").replace(",", ".");
  const n = Number(s);
  return Number.isFinite(n) ? n : null;
}

function parseCzDate(d) {
  const m = String(d ?? "").trim().match(/^(\d{1,2})\.(\d{1,2})\.(\d{4})$/);
  if (!m) return null;
  const day = Number(m[1]);
  const mon = Number(m[2]) - 1;
  const yr = Number(m[3]);
  const dt = new Date(Date.UTC(yr, mon, day));
  if (dt.getUTCFullYear() !== yr || dt.getUTCMonth() !== mon || dt.getUTCDate() !== day) return null;
  return dt;
}

function formatCzDate(dt) {
  if (!dt) return "";
  const d = dt.getUTCDate();
  const m = dt.getUTCMonth() + 1;
  const y = dt.getUTCFullYear();
  return `${String(d).padStart(2,"0")}.${String(m).padStart(2,"0")}.${y}`;
}

function addMonthsUTC(dt, months) {
  const y = dt.getUTCFullYear();
  const m = dt.getUTCMonth();
  const d = dt.getUTCDate();

  const targetMonth = m + months;
  const first = new Date(Date.UTC(y, targetMonth, 1));
  const nextFirst = new Date(Date.UTC(y, targetMonth + 1, 1));
  const lastDay = Math.floor((nextFirst - first) / (24*3600*1000));

  const clampedDay = Math.min(d, lastDay);
  return new Date(Date.UTC(first.getUTCFullYear(), first.getUTCMonth(), clampedDay));
}

function loanPaidSum(loan) {
  const pays = Array.isArray(loan.payments) ? loan.payments : [];
  return pays.reduce((acc, p) => acc + (Number(p.amount) || 0), 0);
}

function loanRemaining(loan) {
  const principal = Number(loan.principal) || 0;
  const paid = loanPaidSum(loan);
  return Math.max(0, principal - paid);
}

function nextDueDate(loan) {
  const first = parseCzDate(loan.firstDate);
  if (!first) return null;

  // pokud nejsou splátky, další termín je 1. splátka
  const pays = Array.isArray(loan.payments) ? loan.payments : [];
  let lastPay = null;
  for (const p of pays) {
    const dt = parseCzDate(p.date);
    if (dt && (!lastPay || dt > lastPay)) lastPay = dt;
  }

  if (!lastPay) return first;

  // další termín = o měsíc po poslední splátce (na stejný den jako firstDate, s ořezem na konec měsíce)
  const day = first.getUTCDate();
  const base = addMonthsUTC(lastPay, 1);
  // nastav den na day v cílovém měsíci (clamp)
  return addMonthsUTC(new Date(Date.UTC(base.getUTCFullYear(), base.getUTCMonth(), day)), 0);
}

function estimatedLastPaymentDate(loan) {
  const remaining = loanRemaining(loan);
  const monthly = Number(loan.monthly) || 0;
  const due = nextDueDate(loan);
  if (!due || monthly <= 0) return null;

  if (remaining <= 0) {
    // už splaceno: vrať datum poslední reálné splátky (pokud existuje), jinak first/due
    const pays = Array.isArray(loan.payments) ? loan.payments : [];
    let lastPay = null;
    for (const p of pays) {
      const dt = parseCzDate(p.date);
      if (dt && (!lastPay || dt > lastPay)) lastPay = dt;
    }
    return lastPay || due;
  }

  const n = Math.ceil(remaining / monthly);
  return addMonthsUTC(due, n - 1);
}

function fillLoansFromData(ls) {
  loans = Array.isArray(ls) ? ls : [];
  loans = loans.map(l => ({
    id: String(l.id || uid()),
    name: String(l.name || ""),
    principal: Number(l.principal) || 0,
    monthly: Number(l.monthly) || 0,
    firstDate: String(l.firstDate || ""),
    payments: Array.isArray(l.payments) ? l.payments.map(p => ({
      date: String(p.date || ""),
      amount: Number(p.amount) || 0
    })) : []
  }));
  renderLoans();
}


function updatePaymentDefaults() {
  const sel = document.getElementById("payLoan");
  const dateInp = document.getElementById("payDate");
  const amtInp = document.getElementById("payAmount");
  if (!sel || !dateInp || !amtInp) return;
  const loan = loans.find(l => l.id === sel.value);
  if (!loan) return;
  const due = nextDueDate(loan);
  if (due && !dateInp.value.trim()) dateInp.value = formatCzDate(due);
  if (!amtInp.value.trim() && loan.monthly) amtInp.value = String(loan.monthly);
}

function renderLoans() {
  const tbody = document.getElementById("loanRows");
  const sel = document.getElementById("payLoan");
  if (!tbody || !sel) return;

  sel.innerHTML = "";
  sel.onchange = updatePaymentDefaults;
  for (const l of loans) {
    const opt = document.createElement("option");
    opt.value = l.id;
    opt.textContent = l.name || "(bez názvu)";
    sel.appendChild(opt);
  }

  tbody.innerHTML = "";
  for (const l of loans) {
    const principal = Number(l.principal) || 0;
    const paid = loanPaidSum(l);
    const rem = loanRemaining(l);
    const monthly = Number(l.monthly) || 0;
    const last = estimatedLastPaymentDate(l);

    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td>${escapeHtml(l.name || "")}</td>
      <td class="num">${czk(principal)}</td>
      <td class="num">${czk(paid)}</td>
      <td class="num">${czk(rem)}</td>
      <td class="num">${monthly ? czk(monthly) : ""}</td>
      <td>${escapeHtml(l.firstDate || "")}</td>
      <td>${escapeHtml(last ? formatCzDate(last) : "")}</td>
      <td>
        <div class="actions">
          <button class="secondary" data-loan-edit="${l.id}">Upravit</button>
          <button class="danger" data-loan-del="${l.id}">Smazat</button>
        </div>
      </td>
    `;
    tbody.appendChild(tr);
  }

  tbody.querySelectorAll("[data-loan-del]").forEach(btn => {
    btn.addEventListener("click", () => {
      const id = btn.getAttribute("data-loan-del");
      loans = loans.filter(x => x.id !== id);
      renderLoans();
    });
  });

  updatePaymentDefaults();

  tbody.querySelectorAll("[data-loan-edit]").forEach(btn => {
    btn.addEventListener("click", () => {
      const id = btn.getAttribute("data-loan-edit");
      const l = loans.find(x => x.id === id);
      if (!l) return;

      const name = prompt("Název úvěru:", l.name || "") ?? l.name;
      const principalStr = prompt("Jistina:", String(l.principal ?? "")) ?? String(l.principal ?? "");
      const monthlyStr = prompt("Měsíční splátka:", String(l.monthly ?? "")) ?? String(l.monthly ?? "");
      const firstDate = prompt("Datum 1. splátky (dd.mm.yyyy):", l.firstDate || "") ?? l.firstDate;

      const principal = parseCzNumber(principalStr);
      const monthly = parseCzNumber(monthlyStr);
      const fd = firstDate ? parseCzDate(firstDate) : null;

      if (name !== null) l.name = name;
      if (principal !== null) l.principal = principal;
      if (monthly !== null) l.monthly = monthly;
      if (firstDate !== null && (firstDate === "" || fd)) l.firstDate = firstDate;

      renderLoans();
    });
  });
}

// ---------- render items ----------
function render() {
  // sync hotovost výpočet
  recalcCash();
  const list = getFiltered();
  const { income, expense, net } = computeSummary(list);

  el("sumIncome").textContent = czk(income);
  el("sumExpense").textContent = czk(expense);
  el("sumNet").textContent = czk(net);
  el("sumCash").textContent = czk(cash.total || 0);

  const tbody = el("rows");
  tbody.innerHTML = "";

  for (const it of list) {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td>${escapeHtml(it.period || "")}</td>
      <td>${escapeHtml(it.date || "")}</td>
      <td>${escapeHtml(it.desc || "")}</td>
      <td class="num">${czk(it.amount)}</td>
      <td>
        <span class="badge ${it.amount >= 0 ? "income" : "expense"}">
          ${it.amount >= 0 ? "příjem" : "výdaj"}
        </span>
      </td>
      <td><span class="badge">${escapeHtml(it.kind || "txn")}</span></td>
      <td>
        <div class="actions">
          <button class="secondary" data-edit="${it.id}">Upravit</button>
          <button class="danger" data-del="${it.id}">Smazat</button>
        </div>
      </td>
    `;
    tbody.appendChild(tr);
  }

  tbody.querySelectorAll("[data-del]").forEach(btn => {
    btn.addEventListener("click", () => {
      const id = btn.getAttribute("data-del");
      items = items.filter(x => x.id !== id);
      render();
    });
  });

  tbody.querySelectorAll("[data-edit]").forEach(btn => {
    btn.addEventListener("click", () => {
      const id = btn.getAttribute("data-edit");
      const it = items.find(x => x.id === id);
      if (!it) return;

      const period = prompt("Období:", it.period || "") ?? it.period;
      const date = prompt("Datum (d.m.rrrr):", it.date || "") ?? it.date;
      const desc = prompt("Popis:", it.desc || "") ?? it.desc;
      const amountStr = prompt("Částka (např. -6750.50):", String(it.amount)) ?? String(it.amount);
      const amount = Number(amountStr);
      if (!Number.isFinite(amount)) { alert("Neplatná částka."); return; }
      const kind = prompt('Kind: "txn", "subtotal", "note":', it.kind || "txn") ?? it.kind;

      it.period = period;
      it.date = date;
      it.desc = desc;
      it.amount = amount;
      it.type = amount >= 0 ? "income" : "expense";
      it.kind = (kind === "subtotal" || kind === "note") ? kind : "txn";

      render();
    });
  });

  el("status").textContent = `Zobrazeno položek: ${list.length} / ${items.length}`;
}

// ---------- API ----------
async function apiGet() {
  const res = await fetch(`${API_URL}?action=get`, { cache: "no-store" });
  if (!res.ok) throw new Error("GET failed");
  return await res.json();
}

async function apiSave(payload) {
  const res = await fetch(`${API_URL}?action=save`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(payload)
  });
  if (!res.ok) throw new Error("SAVE failed");
  return await res.json();
}

// ---------- UI ----------
el("btnLoad").addEventListener("click", async () => {
  try {
    const data = await apiGet();
    items = Array.isArray(data.items) ? data.items : [];
    for (const it of items) {
      if (!it.kind) it.kind = "txn";
    }
    fillCashFromData(data.cash || null);
    fillLoansFromData(data.loans || []);
    render();
  } catch {
    alert("Nepodařilo se načíst data. Zkontroluj api.php a práva k souboru data.json.");
  }
});

el("btnSave").addEventListener("click", async () => {
  try {
    recalcCash();
    await apiSave({ items, cash, loans });
    alert("Uloženo.");
  } catch {
    alert("Nepodařilo se uložit data. Zkontroluj api.php a práva k souboru data.json.");
  }
});

el("btnClear").addEventListener("click", () => {
  if (!confirm("Opravdu smazat všechny položky v aplikaci?")) return;
  items = [];
  loans = [];
  fillCashFromData({ denoms: {}, total: 0 });
  renderLoans();
  render();
});

el("btnAdd").addEventListener("click", () => {
  const period = el("addPeriod").value.trim();
  const date = el("addDate").value.trim();
  const desc = el("addDesc").value.trim();
  const amount = Number(el("addAmount").value.trim());
  const type = el("addType").value;
  const kind = el("addKind").value;

  if (!Number.isFinite(amount)) {
    alert("Zadej číselnou částku (např. 6750 nebo -6750).");
    return;
  }

  items.push({
    id: uid(),
    period,
    date,
    desc: desc || "Bez popisu",
    amount: type === "income" ? Math.abs(amount) : -Math.abs(amount),
    type,
    kind: (kind === "subtotal" || kind === "note") ? kind : "txn"
  });

  el("addPeriod").value = "";
  el("addDate").value = "";
  el("addDesc").value = "";
  el("addAmount").value = "";
  el("addType").value = "expense";
  el("addKind").value = "txn";

  render();
});

el("filterText").addEventListener("input", render);
el("filterType").addEventListener("change", render);

// hotovost listeners
for (const inp of document.querySelectorAll(".cashCount")) {
  inp.addEventListener("input", recalcCash);
  inp.addEventListener("change", recalcCash);
  inp.addEventListener("keyup", recalcCash);
  inp.addEventListener("paste", () => setTimeout(recalcCash, 0));
}
recalcCash();
// úvěry handlers
document.getElementById("btnAddLoan")?.addEventListener("click", () => {
  const name = document.getElementById("loanName").value.trim();
  const principal = parseCzNumber(document.getElementById("loanPrincipal").value);
  const monthly = parseCzNumber(document.getElementById("loanMonthly").value);
  const firstDate = document.getElementById("loanFirstDate").value.trim();

  if (!name) { alert("Zadej název úvěru."); return; }
  if (principal === null || principal <= 0) { alert("Zadej platnou jistinu."); return; }
  if (monthly === null || monthly <= 0) { alert("Zadej platnou měsíční splátku."); return; }
  if (!parseCzDate(firstDate)) { alert("Zadej datum 1. splátky ve formátu dd.mm.yyyy"); return; }

  loans.push({
    id: uid(),
    name,
    principal,
    monthly,
    firstDate,
    payments: []
  });

  document.getElementById("loanName").value = "";
  document.getElementById("loanPrincipal").value = "";
  document.getElementById("loanMonthly").value = "";
  document.getElementById("loanFirstDate").value = "";

  renderLoans();
});

document.getElementById("btnAddPayment")?.addEventListener("click", () => {
  const loanId = document.getElementById("payLoan").value;
  const amount = parseCzNumber(document.getElementById("payAmount").value);
  const date = document.getElementById("payDate").value.trim();

  const loan = loans.find(l => l.id === loanId);
  if (!loan) { alert("Vyber úvěr."); return; }
  if (amount === null || amount <= 0) { alert("Zadej platnou částku splátky."); return; }
  if (!parseCzDate(date)) { alert("Zadej datum splátky ve formátu dd.mm.yyyy"); return; }

  loan.payments = Array.isArray(loan.payments) ? loan.payments : [];
  loan.payments.push({ date, amount });

  document.getElementById("payAmount").value = "";
  document.getElementById("payDate").value = "";

  renderLoans();
});

// initial render
renderLoans();
render();
