<?php
// api.php
// Jednoduché ukládání do JSON souboru v adresáři aplikace.
// Pozn.: Na hostingu musí mít PHP práva zapisovat do data.json.

header("Content-Type: application/json; charset=utf-8");

$action = $_GET["action"] ?? "get";
$file = __DIR__ . DIRECTORY_SEPARATOR . "data.json";

function read_data($file) {
  if (!file_exists($file)) {
    return [
      "items" => [],
      "cash" => ["denoms" => (object)[], "total" => 0],
      "loans" => []
    ];
  }

  $raw = file_get_contents($file);
  if ($raw === false || trim($raw) === "") {
    return [
      "items" => [],
      "cash" => ["denoms" => (object)[], "total" => 0],
      "loans" => []
    ];
  }

  $data = json_decode($raw, true);
  if (!is_array($data)) {
    return [
      "items" => [],
      "cash" => ["denoms" => (object)[], "total" => 0],
      "loans" => []
    ];
  }

  if (!isset($data["items"]) || !is_array($data["items"])) $data["items"] = [];

  if (!isset($data["cash"]) || !is_array($data["cash"])) $data["cash"] = ["denoms" => (object)[], "total" => 0];
  if (!isset($data["cash"]["denoms"]) || !is_array($data["cash"]["denoms"])) $data["cash"]["denoms"] = (object)[];
  if (!isset($data["cash"]["total"]) || !is_numeric($data["cash"]["total"])) $data["cash"]["total"] = 0;

  if (!isset($data["loans"]) || !is_array($data["loans"])) $data["loans"] = [];

  return $data;
}

function write_data($file, $data) {
  $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  if ($json === false) return false;

  $tmp = $file . ".tmp";
  if (file_put_contents($tmp, $json) === false) return false;
  return rename($tmp, $file);
}

if ($action === "get") {
  $data = read_data($file);
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

if ($action === "save") {
  $body = file_get_contents("php://input");
  $payload = json_decode($body, true);

  if (!is_array($payload) || !isset($payload["items"]) || !is_array($payload["items"])) {
    http_response_code(400);
    echo json_encode(["ok" => false, "error" => "Invalid payload"], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // --- items ---
  $cleanItems = [];
  foreach ($payload["items"] as $it) {
    if (!is_array($it)) continue;

    $id = strval($it["id"] ?? "");
    if ($id === "") continue;

    $amount = $it["amount"] ?? 0;
    if (!is_numeric($amount)) continue;
    $amount = floatval($amount);

    $type = ($amount >= 0) ? "income" : "expense";

    $kind = strval($it["kind"] ?? "txn");
    if ($kind !== "txn" && $kind !== "subtotal" && $kind !== "note") $kind = "txn";

    $cleanItems[] = [
      "id" => $id,
      "period" => strval($it["period"] ?? ""),
      "date" => strval($it["date"] ?? ""),
      "desc" => strval($it["desc"] ?? ""),
      "amount" => $amount,
      "type" => $type,
      "kind" => $kind
    ];
  }

  // --- cash ---
  $allowedDenoms = ["1","2","5","10","20","50","100","200","500","1000","2000","5000"];
  $cashPayload = $payload["cash"] ?? ["denoms" => [], "total" => 0];
  if (!is_array($cashPayload)) $cashPayload = ["denoms" => [], "total" => 0];

  $denoms = $cashPayload["denoms"] ?? [];
  if (!is_array($denoms)) $denoms = [];

  $cleanDenoms = [];
  foreach ($allowedDenoms as $d) {
    $v = $denoms[$d] ?? 0;
    $cleanDenoms[$d] = is_numeric($v) ? max(0, intval($v)) : 0;
  }

  $cashTotal = 0;
  foreach ($cleanDenoms as $d => $cnt) {
    $cashTotal += intval($d) * intval($cnt);
  }

  $cleanCash = ["denoms" => $cleanDenoms, "total" => $cashTotal];

  // --- loans ---
  $loansPayload = $payload["loans"] ?? [];
  if (!is_array($loansPayload)) $loansPayload = [];

  $cleanLoans = [];
  foreach ($loansPayload as $l) {
    if (!is_array($l)) continue;
    $id = strval($l["id"] ?? "");
    if ($id === "") continue;

    $payments = $l["payments"] ?? [];
    if (!is_array($payments)) $payments = [];

    $cleanPayments = [];
    foreach ($payments as $p) {
      if (!is_array($p)) continue;
      $amt = $p["amount"] ?? 0;
      if (!is_numeric($amt)) continue;
      $cleanPayments[] = [
        "date" => strval($p["date"] ?? ""),
        "amount" => floatval($amt)
      ];
    }

    $cleanLoans[] = [
      "id" => $id,
      "name" => strval($l["name"] ?? ""),
      "principal" => floatval($l["principal"] ?? 0),
      "monthly" => floatval($l["monthly"] ?? 0),
      "firstDate" => strval($l["firstDate"] ?? ""),
      "payments" => $cleanPayments
    ];
  }

  $data = ["items" => $cleanItems, "cash" => $cleanCash, "loans" => $cleanLoans];
  $ok = write_data($file, $data);

  if (!$ok) {
    http_response_code(500);
    echo json_encode(["ok" => false, "error" => "Failed to write data.json (permissions?)"], JSON_UNESCAPED_UNICODE);
    exit;
  }

  echo json_encode(["ok" => true, "count" => count($cleanItems)], JSON_UNESCAPED_UNICODE);
  exit;
}

http_response_code(400);
echo json_encode(["ok" => false, "error" => "Unknown action"], JSON_UNESCAPED_UNICODE);
