
const BASE = "https://v6.db.transport.rest";

function pickJourney(journeys, preference) {
  if (!Array.isArray(journeys) || journeys.length === 0) return null;
  if (preference === "transfers") {
    return journeys.slice().sort((a, b) => (a.legs?.length || 999) - (b.legs?.length || 999))[0];
  }
  // fastest: keep the first or sort by duration string (rough)
  return journeys[0];
}

function durationToMinutes(duration) {
  const m = /^PT(?:(\d+)H)?(?:(\d+)M)?/i.exec(duration || "");
  if (!m) return null;
  const h = m[1] ? Number(m[1]) : 0;
  const mins = m[2] ? Number(m[2]) : 0;
  return h * 60 + mins;
}

function stopPoint(stop, when) {
  if (!stop) return null;
  const lat = stop.location?.latitude ?? stop.latitude;
  const lon = stop.location?.longitude ?? stop.longitude;
  return {
    name: stop.name || stop.id || "Stop",
    id: stop.id || null,
    latitude: typeof lat === "number" ? lat : null,
    longitude: typeof lon === "number" ? lon : null,
    when: when || null,
  };
}

export async function routeWithDb({ from, to, whenISO, preference }) {
  const url = new URL(BASE + "/journeys");
  url.searchParams.set("from", from);
  url.searchParams.set("to", to);
  url.searchParams.set("departure", whenISO);
  url.searchParams.set("results", "3");
  url.searchParams.set("stopovers", "true");
  url.searchParams.set("remarks", "true");
  url.searchParams.set("language", "cs");

  const res = await fetch(url.toString(), { headers: { "accept": "application/json" } });
  if (!res.ok) {
    const txt = await res.text().catch(() => "");
    throw new Error(`DB API error ${res.status}: ${txt.slice(0, 200)}`);
  }
  const data = await res.json();

  const journey = pickJourney(data.journeys, preference);
  if (!journey) throw new Error("Nenalezeno žádné spojení.");

  const legs = journey.legs || [];

  const transfers = [];
  for (let i = 0; i < legs.length - 1; i++) {
    const a = legs[i]?.destination?.name;
    const b = legs[i + 1]?.origin?.name;
    if (a && b && a === b) transfers.push(a);
  }

  const stopsMap = new Map();
  const addStop = (sp) => {
    if (!sp || typeof sp.latitude !== "number" || typeof sp.longitude !== "number") return;
    const key = `${sp.name}|${sp.latitude.toFixed(5)}|${sp.longitude.toFixed(5)}`;
    if (!stopsMap.has(key)) stopsMap.set(key, sp);
  };

  const segments = legs
    .filter(l => l.origin && l.destination)
    .map(l => {
      const path = [];

      const o = stopPoint(l.origin, l.departure);
      addStop(o); path.push(o);

      if (Array.isArray(l.stopovers)) {
        for (const so of l.stopovers) {
          const sp = stopPoint(so.stop, so.arrival || so.departure || null);
          addStop(sp); path.push(sp);
        }
      }

      const d = stopPoint(l.destination, l.arrival);
      addStop(d); path.push(d);

      return {
        mode: l.mode || null,
        line: l.line?.name || l.name || null,
        operator: l.line?.operator?.name || null,
        from: l.origin?.name || "",
        to: l.destination?.name || "",
        departure: l.departure || null,
        arrival: l.arrival || null,
        path,
      };
    });

  const durationMinutes = durationToMinutes(journey.duration) ?? null;

  return {
    provider: "db",
    from,
    to,
    when: whenISO,
    preference,
    durationMinutes,
    transfers,
    stops: Array.from(stopsMap.values()),
    segments,
  };
}
