import express from "express";
import cors from "cors";
import "dotenv/config";
import path from "path";
import { fileURLToPath } from "url";

import { routeWithDb } from "./providers/db.js";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();

// If frontend is served by the same app, CORS is not required, but harmless.
app.use(cors());
app.use(express.json({ limit: "1mb" }));

// Serve the frontend
app.use(express.static(path.join(__dirname, "public")));

app.get("/health", (req, res) => res.json({ ok: true }));

app.get("/route", async (req, res) => {
  try {
    const from = String(req.query.from || "").trim();
    const to = String(req.query.to || "").trim();
    const when = String(req.query.when || "").trim(); // ISO
    const preference = String(req.query.preference || "fastest");

    if (!from || !to || !when) {
      return res.status(400).json({ error: "Chybí parametry: from, to, when (ISO)." });
    }

    const itinerary = await routeWithDb({ from, to, whenISO: when, preference });
    res.json(itinerary);
  } catch (e) {
    res.status(500).json({ error: e?.message || String(e) });
  }
});

// Fallback: serve index for any other path (nice on shared hosting routes)
app.get("*", (req, res) => {
  res.sendFile(path.join(__dirname, "public", "index.html"));
});

const port = Number(process.env.PORT || 8787);
app.listen(port, () => console.log(`Server running on http://localhost:${port}`));
