
const HUB_NAME = "Hradec Králové";
const SERVER = "http://localhost:8787";

const map = L.map("map", { zoomControl: true, worldCopyJump: true });
L.tileLayer("https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png", {
  maxZoom: 19,
  attribution: "&copy; OpenStreetMap přispěvatelé",
}).addTo(map);

const europeBounds = L.latLngBounds(L.latLng(35, -12), L.latLng(66, 35));
map.fitBounds(europeBounds, { padding: [20, 20] });

const routeLayer = L.layerGroup().addTo(map);
const markerLayer = L.layerGroup().addTo(map);

function setMsg(msg, ok=null) {
  const el = document.getElementById("statusMsg");
  el.textContent = msg;
  el.style.borderColor = ok === null ? "#eee" : (ok ? "#cfe9cf" : "#f1c7c7");
  el.style.background = ok === null ? "#fafafa" : (ok ? "#f2fbf2" : "#fff5f5");
}

async function pingServer() {
  try {
    const r = await fetch(`${SERVER}/health`);
    const t = await r.json();
    document.getElementById("serverState").textContent = t.ok ? "OK" : "chyba";
  } catch {
    document.getElementById("serverState").textContent = "nedostupný";
  }
}

function clearRoute() {
  routeLayer.clearLayers();
  markerLayer.clearLayers();
  setMsg("Vyčištěno.", null);
}

function dot(kind) {
  const fill = kind === "hub" ? "#d32f2f" : (kind === "transfer" ? "#ef6c00" : "#1565c0");
  const r = kind === "hub" ? 8 : (kind === "transfer" ? 6 : 5);
  return { radius: r, color: "#fff", weight: 2, fillColor: fill, fillOpacity: 0.95 };
}

function tip(html) {
  return { className: "custom-tip", sticky: true, direction: "top" , opacity: 1};
}

function fmtTime(iso) {
  if (!iso) return "";
  const d = new Date(iso);
  return d.toLocaleString("cs-CZ", { hour: "2-digit", minute: "2-digit", year: "numeric", month: "2-digit", day: "2-digit" });
}

function renderItinerary(data) {
  clearRoute();

  const stops = data.stops || [];
  const transfers = new Set(data.transfers || []);
  const coords = [];

  for (const s of stops) {
    if (typeof s.latitude !== "number" || typeof s.longitude !== "number") continue;
    coords.push([s.latitude, s.longitude]);

    const isHub = (s.name || "").toLowerCase().includes("hradec");
    const isTransfer = transfers.has(s.name);
    const kind = isHub ? "hub" : (isTransfer ? "transfer" : "stop");

    L.circleMarker([s.latitude, s.longitude], dot(kind))
      .addTo(markerLayer)
      .bindTooltip(`<b>${s.name}</b>${s.when ? "<br/>" + fmtTime(s.when) : ""}`, tip());
  }

  for (const seg of (data.segments || [])) {
    const pts = (seg.path || [])
      .filter(p => typeof p.latitude === "number" && typeof p.longitude === "number")
      .map(p => [p.latitude, p.longitude]);

    if (pts.length >= 2) {
      L.polyline(pts, { color: "#1565c0", weight: 4, opacity: 0.75 }).addTo(routeLayer)
        .bindTooltip(
          `<b>${seg.line || "Úsek"}</b><br/>${seg.from} → ${seg.to}<br/>${fmtTime(seg.departure)} → ${fmtTime(seg.arrival)}`,
          tip()
        );
    }
  }

  if (coords.length) map.fitBounds(L.latLngBounds(coords), { padding: [30, 30] });

  setMsg(`Nalezeno: ${data.to} • ${data.durationMinutes} min • přestupy: ${data.transfers.length}`, true);
}

const whenEl = document.getElementById("when");
const prefEl = document.getElementById("preference");
const searchEl = document.getElementById("search");
const listEl = document.getElementById("cityList");

document.getElementById("clear").addEventListener("click", clearRoute);
document.getElementById("fit").addEventListener("click", () => map.fitBounds(europeBounds, { padding: [20,20] }));

function ensureWhen() {
  if (whenEl.value) return whenEl.value;
  const d = new Date();
  d.setDate(d.getDate() + 1);
  d.setHours(8,0,0,0);
  const pad = n => String(n).padStart(2,"0");
  const v = `${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())}T${pad(d.getHours())}:${pad(d.getMinutes())}`;
  whenEl.value = v;
  return v;
}

async function fetchRoute(toCity) {
  const whenLocal = ensureWhen();
  const preference = prefEl.value;
  setMsg(`Hledám spojení: ${HUB_NAME} → ${toCity} (${whenLocal})…`, null);

  try {
    const url = new URL(`${SERVER}/route`);
    url.searchParams.set("from", HUB_NAME);
    url.searchParams.set("to", toCity);
    url.searchParams.set("when", new Date(whenLocal).toISOString());
    url.searchParams.set("preference", preference);

    const res = await fetch(url.toString());
    const data = await res.json();

    if (!res.ok) {
      setMsg(`Chyba: ${data.error || res.statusText}`, false);
      return;
    }

    renderItinerary(data);
  } catch (e) {
    setMsg(`Server nedostupný. Spusť backend: npm start (chyba: ${e?.message || e})`, false);
  }
}

function renderList() {
  const q = (searchEl.value || "").trim().toLowerCase();
  const items = (window.CAPITALS || []).filter(c =>
    c.capital.toLowerCase().includes(q) || c.country.toLowerCase().includes(q)
  );

  listEl.innerHTML = "";
  for (const c of items) {
    const div = document.createElement("div");
    div.className = "item";
    div.innerHTML = `
      <div>
        <div class="name">${c.flag} ${c.capital}</div>
        <div class="meta">${c.country}</div>
      </div>
      <div class="badge">najít</div>
    `;
    div.addEventListener("click", () => fetchRoute(c.capital));
    listEl.appendChild(div);
  }
}

searchEl.addEventListener("input", renderList);
ensureWhen();
renderList();
pingServer();
