<?php
include 'auth.php';
include 'config.php';
include 'helpers.php';

$file = 'data.json';
if (!file_exists($file)) file_put_contents($file, '[]');
$data = json_decode(file_get_contents($file), true);
if (!is_array($data)) $data = [];

// --- Filtry (GET) ---
$q = trim((string)($_GET['q'] ?? ''));
$sklad = trim((string)($_GET['sklad'] ?? ''));
$statusFilter = $_GET['status'] ?? 'all'; // all|aktivni|prodano
$platFilter = $_GET['platformy'] ?? [];
if (!is_array($platFilter)) $platFilter = [$platFilter];
$usePlatFilter = count($platFilter) > 0;

$sort = $_GET['sort'] ?? 'newest'; // newest|nazev|cena|status|sklad
$dir = $_GET['dir'] ?? 'desc'; // asc|desc
$dir = ($dir === 'asc') ? 'asc' : 'desc';

// --- Výpočty pro limity + součty ---
$activeCount = [];
$totalCount = [];
$totalValueAll = 0.0;
$totalValueActive = 0.0;
$totalValueSold = 0.0;

foreach ($data as $d) {
  $st = norm_status($d['status'] ?? 'aktivni');
  $price = (float)($d['cena'] ?? 0);
  $platformy = ensure_array($d['platformy'] ?? []);

  $totalValueAll += $price;
  if ($st === 'aktivni') $totalValueActive += $price;
  else $totalValueSold += $price;

  foreach ($platformy as $p) {
    $totalCount[$p] = ($totalCount[$p] ?? 0) + 1;
    if ($st === 'aktivni') $activeCount[$p] = ($activeCount[$p] ?? 0) + 1;
  }
}

function limit_class($platform, $activeCount, $LIMITS){
  if (!isset($LIMITS[$platform])) return '';
  $limit = (int)$LIMITS[$platform];
  $c = (int)($activeCount[$platform] ?? 0);
  if ($limit <= 0) return '';
  $ratio = $c / $limit;
  if ($c >= $limit) return 'limit-over';
  if ($ratio >= 0.9) return 'limit-warn';
  return 'limit-ok';
}

// --- Aplikovat filtry na seznam ---
$filtered = [];
foreach ($data as $i => $d) {
  $st = norm_status($d['status'] ?? 'aktivni');
  if ($statusFilter !== 'all' && $st !== $statusFilter) continue;

  $name = (string)($d['nazev'] ?? '');
  $note = (string)($d['poznamka'] ?? '');
  if ($q !== '' && !str_contains_ci($name, $q)) continue;
  if ($sklad !== '' && !str_contains_ci($note, $sklad)) continue;

  $plats = ensure_array($d['platformy'] ?? []);
  if ($usePlatFilter && !intersects($plats, $platFilter)) continue;

  $d['_id'] = $i; // stabilní odkaz na edit/smazat
  $filtered[] = $d;
}

// --- Řazení ---
usort($filtered, function($a, $b) use ($sort, $dir) {
  $mul = ($dir === 'asc') ? 1 : -1;

  if ($sort === 'nazev') {
    return $mul * strcmp(mb_strtolower((string)($a['nazev'] ?? ''), 'UTF-8'), mb_strtolower((string)($b['nazev'] ?? ''), 'UTF-8'));
  }
  if ($sort === 'cena') {
    $aa = (float)($a['cena'] ?? 0); $bb = (float)($b['cena'] ?? 0);
    if ($aa == $bb) return 0;
    return ($aa < $bb ? -1 : 1) * $mul;
  }
  if ($sort === 'status') {
    $aa = (string)($a['status'] ?? 'aktivni'); $bb = (string)($b['status'] ?? 'aktivni');
    return $mul * strcmp($aa, $bb);
  }
  if ($sort === 'sklad') {
    return $mul * strcmp(mb_strtolower((string)($a['poznamka'] ?? ''), 'UTF-8'), mb_strtolower((string)($b['poznamka'] ?? ''), 'UTF-8'));
  }
  // newest (výchozí) - podle _id (pořadí vložení)
  return $mul * ((int)$a['_id'] - (int)$b['_id']);
});

function build_query($overrides = []){
  $q = $_GET;
  foreach($overrides as $k=>$v){
    if ($v === null) unset($q[$k]);
    else $q[$k] = $v;
  }
  $parts = [];
  foreach($q as $k=>$v){
    if (is_array($v)) {
      foreach($v as $vv) $parts[] = urlencode($k.'[]').'='.urlencode($vv);
    } else {
      $parts[] = urlencode($k).'='.urlencode($v);
    }
  }
  return implode('&', $parts);
}

function sort_link($key){
  $currentSort = $_GET['sort'] ?? 'newest';
  $currentDir = $_GET['dir'] ?? 'desc';
  $dir = 'asc';
  if ($currentSort === $key && $currentDir === 'asc') $dir = 'desc';
  return 'index.php?'.build_query(['sort'=>$key,'dir'=>$dir]);
}

?>
<!doctype html>
<html lang="cs">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<link rel="stylesheet" href="style.css">
<title>Inzeráty – Jarda</title>
</head>
<body>
<header class="topbar">
  <div>
    <h1>Evidence inzerátů – Jarda</h1>
    <div class="muted">Nově: filtrování + řazení seznamu. Poznámka = kde je věc uskladněná.</div>
  </div>
  <div class="actions">
    <a class="btn" href="import.php">Import CSV</a>
    <a class="btn" href="print.php" target="_blank">Tiskový přehled</a>
  </div>
</header>

<section class="card">
  <h2>Součet hodnoty položek</h2>
  <div class="stats">
    <div class="stat"><div class="stat-k">Celkem</div><div class="stat-v"><?=number_format($totalValueAll, 0, ',', ' ')?> Kč</div></div>
    <div class="stat"><div class="stat-k">Aktivní</div><div class="stat-v"><?=number_format($totalValueActive, 0, ',', ' ')?> Kč</div></div>
    <div class="stat"><div class="stat-k">Prodáno</div><div class="stat-v"><?=number_format($totalValueSold, 0, ',', ' ')?> Kč</div></div>
  </div>
</section>

<section class="card">
  <h2>Přidat inzerát</h2>
  <form method="post" action="save.php" class="grid" enctype="multipart/form-data">
    <div>
      <label>Název</label>
      <input name="nazev" required>
    </div>
    <div>
      <label>Cena (Kč)</label>
      <input name="cena" type="number" step="1" min="0" required>
    </div>
    <div>
      <label>Stav</label>
      <select name="status">
        <option value="aktivni" selected>aktivní</option>
        <option value="prodano">prodáno</option>
      </select>
    </div>
    <div>
      <label>Platformy</label>
      <div class="checks">
        <?php foreach($PLATFORMS as $k=>$v): ?>
          <label class="check"><input type="checkbox" name="platformy[]" value="<?=h($k)?>"> <?=h($v)?></label>
        <?php endforeach; ?>
      </div>
    </div>
    <div class="full">
      <label>Kde je uskladněno (poznámka)</label>
      <input name="poznamka" placeholder="např. sklep, garáž, regál 3, krabice A">
    </div>
    <div class="full">
      <label>Fotky (více souborů)</label>
      <input type="file" name="fotky[]" accept="image/*" multiple>
    </div>
    <div class="full">
      <button class="btn primary">Přidat</button>
      <span class="muted">Prodáno se nepočítá do limitu.</span>
    </div>
  </form>
</section>

<section class="card">
  <h2>Limity platforem</h2>
  <div class="limits">
    <?php foreach ($LIMITS as $k => $l):
      $c = (int)($activeCount[$k] ?? 0);
      $t = (int)($totalCount[$k] ?? 0);
      $cls = limit_class($k, $activeCount, $LIMITS);
    ?>
      <div class="limitbox <?=$cls?>">
        <div class="limit-title"><?=h($PLATFORMS[$k] ?? $k)?></div>
        <div class="limit-main"><b><?=$c?></b> / <?=$l?> <span class="muted">(aktivní)</span></div>
        <div class="muted small">celkem vazeb: <?=$t?></div>
      </div>
    <?php endforeach; ?>
  </div>
</section>

<section class="card">
  <h2>Filtr & řazení</h2>
  <form method="get" class="grid">
    <div>
      <label>Hledat název</label>
      <input name="q" value="<?=h($q)?>" placeholder="např. iPhone, boty...">
    </div>
    <div>
      <label>Uskladnění (poznámka obsahuje)</label>
      <input name="sklad" value="<?=h($sklad)?>" placeholder="např. sklep">
    </div>
    <div>
      <label>Stav</label>
      <select name="status">
        <option value="all" <?=$statusFilter==='all'?'selected':''?>>vše</option>
        <option value="aktivni" <?=$statusFilter==='aktivni'?'selected':''?>>aktivní</option>
        <option value="prodano" <?=$statusFilter==='prodano'?'selected':''?>>prodáno</option>
      </select>
    </div>
    <div>
      <label>Řazení</label>
      <select name="sort">
        <option value="newest" <?=$sort==='newest'?'selected':''?>>nejnovější</option>
        <option value="nazev" <?=$sort==='nazev'?'selected':''?>>název</option>
        <option value="cena" <?=$sort==='cena'?'selected':''?>>cena</option>
        <option value="status" <?=$sort==='status'?'selected':''?>>stav</option>
        <option value="sklad" <?=$sort==='sklad'?'selected':''?>>uskladnění</option>
      </select>
      <div class="muted small">
        <label class="check" style="margin-top:6px"><input type="checkbox" name="dir" value="asc" <?=$dir==='asc'?'checked':''?>> vzestupně</label>
      </div>
    </div>

    <div class="full">
      <label>Platformy (když nic nezaškrtneš, bere se vše)</label>
      <div class="checks">
        <?php foreach($PLATFORMS as $k=>$v): ?>
          <label class="check">
            <input type="checkbox" name="platformy[]" value="<?=h($k)?>" <?=in_array($k,$platFilter,true)?'checked':''?>>
            <?=h($v)?>
          </label>
        <?php endforeach; ?>
      </div>
    </div>

    <div class="full">
      <button class="btn primary">Použít</button>
      <a class="btn" href="index.php">Reset</a>
      <span class="muted">Zobrazeno: <b><?=count($filtered)?></b> / <?=count($data)?></span>
    </div>
  </form>
</section>

<section class="card">
  <h2>Seznam inzerátů</h2>
  <div class="muted small">Rychlé řazení klikem: <a class="link" href="<?=h(sort_link('nazev'))?>">název</a> · <a class="link" href="<?=h(sort_link('cena'))?>">cena</a> · <a class="link" href="<?=h(sort_link('status'))?>">stav</a> · <a class="link" href="<?=h(sort_link('sklad'))?>">uskladnění</a></div>
  <div class="table-wrap">
    <table>
      <thead>
        <tr><th>Galerie</th><th>Název</th><th>Cena</th><th>Platformy</th><th>Stav</th><th>Uskladnění</th><th>Akce</th></tr>
      </thead>
      <tbody>
      <?php foreach ($filtered as $d):
        $id = (int)$d['_id'];
        $st = norm_status($d['status'] ?? 'aktivni');
        $stLabel = $st === 'prodano' ? 'prodáno' : 'aktivní';
        $stCls = $st === 'prodano' ? 'status-sold' : 'status-active';
        $platformy = ensure_array($d['platformy'] ?? []);
        $fotky = ensure_array($d['fotky'] ?? []);
      ?>
        <tr>
          <td>
            <?php if (count($fotky) > 0): ?>
              <div class="gallery">
                <?php foreach($fotky as $f): ?>
                  <a class="gitem" href="uploads/<?=h($f)?>" target="_blank"><img class="thumb" src="uploads/<?=h($f)?>" alt=""></a>
                <?php endforeach; ?>
              </div>
            <?php else: ?><span class="muted small">–</span><?php endif; ?>
          </td>
          <td><?=h($d['nazev'] ?? '')?></td>
          <td><?=number_format((float)($d['cena'] ?? 0), 0, ',', ' ')?></td>
          <td><?=h(platforms_to_string($platformy))?></td>
          <td><span class="pill <?=$stCls?>"><?=$stLabel?></span></td>
          <td><?=h($d['poznamka'] ?? '')?></td>
          <td>
            <a href="edit.php?id=<?=$id?>" class="link">Edit</a>
            <span class="sep">|</span>
            <a href="delete.php?id=<?=$id?>" class="link danger" onclick="return confirm('Opravdu smazat?');">Smazat</a>
          </td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</section>

<footer class="footer muted small">Ukládáno do <code>data.json</code>, fotky v <code>/uploads</code></footer>
</body>
</html>
