// Horliko podlahářství – interactivity (menu, reveal, form validation, availability)

const $ = (sel, root = document) => root.querySelector(sel);
const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

function initYear() {
  const el = $("#year");
  if (el) el.textContent = String(new Date().getFullYear());
}

function initNav() {
  const toggle = $("#navToggle");
  const nav = $("#siteNav");
  if (!toggle || !nav) return;

  const setOpen = (open) => {
    nav.classList.toggle("is-open", open);
    toggle.setAttribute("aria-expanded", String(open));
    document.body.classList.toggle("nav-open", open);
  };

  toggle.addEventListener("click", () => {
    const isOpen = nav.classList.contains("is-open");
    setOpen(!isOpen);
  });

  // Close after clicking a link in the menu
  nav.addEventListener("click", (e) => {
    const a = e.target.closest("a");
    if (a) setOpen(false);
  });

  // Close on ESC
  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape") setOpen(false);
  });
}

function initReveal() {
  const els = $$(".reveal");
  if (!els.length) return;

  if (!("IntersectionObserver" in window)) {
    els.forEach((el) => el.classList.add("is-visible"));
    return;
  }

  const io = new IntersectionObserver(
    (entries) => {
      for (const entry of entries) {
        if (entry.isIntersecting) {
          entry.target.classList.add("is-visible");
          io.unobserve(entry.target);
        }
      }
    },
    { threshold: 0.12 }
  );

  els.forEach((el) => io.observe(el));
}

function validateEmail(email) {
  return /^[^\s@]+@[^\s@]+\.[^\s@]{2,}$/.test(String(email).trim());
}

function setFieldError(name, msg) {
  const el = document.querySelector(`.error[data-for="${name}"]`);
  if (el) el.textContent = msg || "";
}

function initSuccessModal() {
  const modal = $("#successModal");
  if (!modal) {
    // Provide no-op so initForm can call it safely even if modal is removed.
    window.openSuccessModal = () => {};
    return;
  }

  const backdrop = modal.querySelector(".modal__backdrop");
  const closeBtn = $("#modalClose");

  const open = () => {
    modal.classList.add("is-open");
    modal.setAttribute("aria-hidden", "false");
    document.body.classList.add("modal-lock");
  };

  const close = () => {
    modal.classList.remove("is-open");
    modal.setAttribute("aria-hidden", "true");
    document.body.classList.remove("modal-lock");
  };

  // Expose for initForm (keeps code simple)
  window.openSuccessModal = open;

  if (backdrop) backdrop.addEventListener("click", close);
  if (closeBtn) closeBtn.addEventListener("click", close);

  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape" && modal.classList.contains("is-open")) close();
  });
}

function initForm() {
  const form = $("#contactForm");
  if (!form) return;

  // Form is submitted to a hidden iframe so the user stays on the page.
  // This lets us show a success popup even on simple hosting without a backend.
  const iframe = document.getElementById("hidden_iframe");
  let submittedToIframe = false;

  const status = $("#formStatus");
  const setStatus = (msg) => {
    if (status) status.textContent = msg || "";
  };

  const getValue = (id) => (document.getElementById(id)?.value ?? "").trim();

  const validate = () => {
    let ok = true;
    ["name", "email", "service", "message", "gdpr"].forEach((k) => setFieldError(k, ""));

    const name = getValue("name");
    const email = getValue("email");
    const service = getValue("service");
    const message = getValue("message");
    const gdpr = document.getElementById("gdpr")?.checked;

    if (name.length < 2) {
      setFieldError("name", "Zadejte prosím jméno.");
      ok = false;
    }
    if (!validateEmail(email)) {
      setFieldError("email", "Zadejte prosím platný e-mail.");
      ok = false;
    }
    if (!service) {
      setFieldError("service", "Vyberte prosím službu.");
      ok = false;
    }
    if (message.length < 10) {
      setFieldError("message", "Popište prosím zakázku (alespoň 10 znaků).");
      ok = false;
    }
    if (!gdpr) {
      setFieldError("gdpr", "Souhlas je povinný.");
      ok = false;
    }
    // Přílohy – volitelné, ale hlídáme typy a velikost (celkem max 5 MB)
    const filesInput = document.getElementById("attachments");
    if (filesInput && filesInput.files && filesInput.files.length) {
      const maxBytes = 5 * 1024 * 1024;
      let total = 0;
      const allowed = ["image/jpeg", "image/png", "application/pdf"];
      let badType = false;

      for (const f of filesInput.files) {
        total += f.size || 0;
        if (f.type && !allowed.includes(f.type)) badType = true;
      }

      if (badType) {
        setFieldError("attachments", "Povoleny jsou pouze JPG, PNG nebo PDF.");
        ok = false;
      } else if (total > maxBytes) {
        setFieldError("attachments", "Celková velikost příloh může být max. 5 MB.");
        ok = false;
      }
    }


    return ok;
  };

  // If the hidden iframe loads after a submit, treat it as success.
  // (FormSubmit returns some HTML in the iframe; the main page stays intact.)
  if (iframe) {
    iframe.addEventListener("load", () => {
      if (!submittedToIframe) return;

      submittedToIframe = false;
      setStatus("");
      try {
        if (typeof window.openSuccessModal === "function") window.openSuccessModal();
      } catch (_) {}
      try { form.reset(); } catch (_) {}
    });
  }

  // Po návratu z odeslání zobraz potvrzení (FormSubmit přesměruje na _next)
  const params = new URLSearchParams(window.location.search);
  if (params.get("sent") === "1") {
    setStatus("");
    try {
      if (typeof window.openSuccessModal === "function") window.openSuccessModal();
    } catch (_) {
      // ignore
    }

    // Uklid URL (ať se modal neotvírá po refreshi)
    params.delete("sent");
    const clean = window.location.pathname + (params.toString() ? "?" + params.toString() : "") + "#kontakt";
    window.history.replaceState({}, "", clean);
  }

  form.addEventListener("submit", (e) => {
    if (!validate()) {
      e.preventDefault();
      setStatus("Zkontrolujte prosím vyplnění polí.");
      return;
    }

    // Mark that the next iframe load should be treated as the "sent" callback.
    submittedToIframe = true;

    // Keep old redirect-based flow as a fallback (e.g., if target iframe is removed).
    const next = form.querySelector('input[name="_next"]');
    if (next) {
      const url = new URL(window.location.href);
      url.searchParams.set("sent", "1");
      url.hash = "kontakt";
      next.value = url.toString();
    }

    setStatus("Odesílám…");
  });
}

async function applyAvailability() {
  const el = document.getElementById("availability");
  const textEl = document.getElementById("availabilityText");
  if (!el) return;

  // 1) Prefer server-side status (global for all visitors)
  try {
    const res = await fetch("api/get.php?t=" + Date.now(), { cache: "no-store" });
    if (res.ok) {
      const data = await res.json();
      if (data && data.status) {
        el.classList.remove("available", "limited", "unavailable");
        el.classList.add(data.status);
        if (textEl && data.text) textEl.textContent = data.text;
        return;
      }
    }
  } catch (_) {
    // ignore
  }

  // 2) Fallback: localStorage (only for this device)
  const key = "horliko_availability";
  let stored = null;
  try {
    stored = JSON.parse(localStorage.getItem(key) || "null");
  } catch (_) {
    stored = null;
  }

  if (stored && stored.status) {
    el.classList.remove("available", "limited", "unavailable");
    el.classList.add(stored.status);
    if (textEl && stored.text) textEl.textContent = stored.text;
  }
}

// Enable CSS that depends on JS (animations, reveal)
document.documentElement.classList.add("js");

// Boot
initYear();
initNav();
initReveal();
initSuccessModal();
initForm();
applyAvailability();
