<?php
// Horliko podlahářství – jednoduchý backend pro odeslání poptávky.
// Funguje na běžném webhostingu s PHP (bez databáze).

header('Content-Type: application/json; charset=utf-8');
header('X-Content-Type-Options: nosniff');

// ====== Nastavení ======
$TO_EMAIL = 'hornik34@seznam.cz';  // kam mají chodit poptávky
$MAX_TOTAL_BYTES = 5 * 1024 * 1024; // 5 MB celkem
$ALLOWED_MIME = [
  'image/jpeg' => 'jpg',
  'image/png'  => 'png',
  'application/pdf' => 'pdf',
];
$MAX_FILES = 6;

function respond($ok, $error = null, $http = 200) {
  http_response_code($http);
  $out = ['ok' => (bool)$ok];
  if (!$ok && $error) $out['error'] = $error;
  echo json_encode($out, JSON_UNESCAPED_UNICODE);
  exit;
}

// Pouze POST
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  respond(false, 'Neplatná metoda.', 405);
}

// Honeypot proti spamu
if (!empty($_POST['website'] ?? '')) {
  // Tichý úspěch (spam boti nepoznají)
  respond(true);
}

// Základní pole
$name    = trim((string)($_POST['name'] ?? ''));
$email   = trim((string)($_POST['email'] ?? ''));
$phone   = trim((string)($_POST['phone'] ?? ''));
$service = trim((string)($_POST['service'] ?? ''));
$message = trim((string)($_POST['message'] ?? ''));
$gdpr    = isset($_POST['gdpr']);

if (mb_strlen($name) < 2) respond(false, 'Vyplňte prosím jméno.', 400);
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) respond(false, 'Vyplňte prosím platný e-mail.', 400);
if ($service === '') respond(false, 'Vyberte prosím službu.', 400);
if (mb_strlen($message) < 10) respond(false, 'Popište prosím zakázku (alespoň 10 znaků).', 400);
if (!$gdpr) respond(false, 'Souhlas je povinný.', 400);

// Bezpečné hodnoty do HTML
$esc = function($s) {
  return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
};

$now = new DateTime('now', new DateTimeZone('Europe/Prague'));
$submittedAt = $now->format('j.n.Y H:i');

$htmlMessage = '';
$htmlMessage .= '<div style="font-family:Arial,Helvetica,sans-serif;font-size:14px;line-height:1.45;color:#111">';
$htmlMessage .= '<h2 style="margin:0 0 12px 0">Nová poptávka – Horliko podlahářství</h2>';
$htmlMessage .= '<table cellpadding="0" cellspacing="0" style="border-collapse:collapse;width:100%;max-width:720px">';

$row = function($label, $valueHtml) {
  return '<tr>'
    . '<td style="padding:8px 10px;border:1px solid #e5e7eb;background:#f9fafb;width:180px;vertical-align:top"><strong>' . $label . '</strong></td>'
    . '<td style="padding:8px 10px;border:1px solid #e5e7eb;vertical-align:top">' . $valueHtml . '</td>'
    . '</tr>';
};

$htmlMessage .= $row('Jméno', $esc($name));
$htmlMessage .= $row('E-mail', '<a href="mailto:' . $esc($email) . '">' . $esc($email) . '</a>');
$htmlMessage .= $row('Telefon', $phone !== '' ? $esc($phone) : '<span style="color:#6b7280">neuvedeno</span>');
$htmlMessage .= $row('Služba', $esc($service));
$htmlMessage .= $row('Odesláno', $esc($submittedAt));
$htmlMessage .= $row('Zpráva', nl2br($esc($message)));
$htmlMessage .= '</table>';
$htmlMessage .= '<p style="margin:12px 0 0 0;color:#6b7280">Tato zpráva byla odeslána z webového formuláře.</p>';
$htmlMessage .= '</div>';

// Přílohy
$attachments = [];
$totalBytes = 0;

if (!empty($_FILES['attachments']) && is_array($_FILES['attachments']['name'])) {
  $names = $_FILES['attachments']['name'];
  $types = $_FILES['attachments']['type'];
  $tmp   = $_FILES['attachments']['tmp_name'];
  $errs  = $_FILES['attachments']['error'];
  $sizes = $_FILES['attachments']['size'];

  $count = min(count($names), $MAX_FILES);

  for ($i = 0; $i < $count; $i++) {
    if (($errs[$i] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) continue;
    if (($errs[$i] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) {
      respond(false, 'Nepodařilo se nahrát přílohu. Zkuste to prosím znovu.', 400);
    }

    $size = (int)($sizes[$i] ?? 0);
    $totalBytes += $size;
    if ($totalBytes > $MAX_TOTAL_BYTES) {
      respond(false, 'Celková velikost příloh může být max. 5 MB.', 400);
    }

    $mime = (string)($types[$i] ?? '');

    // Spolehlivější detekce typu
    if (function_exists('finfo_open')) {
      $finfo = finfo_open(FILEINFO_MIME_TYPE);
      if ($finfo) {
        $detected = finfo_file($finfo, $tmp[$i]);
        finfo_close($finfo);
        if ($detected) $mime = $detected;
      }
    }

    if (!isset($ALLOWED_MIME[$mime])) {
      respond(false, 'Povoleny jsou pouze JPG, PNG nebo PDF.', 400);
    }

    $originalName = (string)($names[$i] ?? 'soubor');
    $safeName = preg_replace('/[^a-zA-Z0-9._-]+/', '_', $originalName);
    if ($safeName === '' || $safeName === '_' ) $safeName = 'soubor.' . $ALLOWED_MIME[$mime];

    $content = @file_get_contents($tmp[$i]);
    if ($content === false) {
      respond(false, 'Nepodařilo se přečíst přílohu.', 400);
    }

    $attachments[] = [
      'name' => $safeName,
      'mime' => $mime,
      'data' => $content,
    ];
  }
}

// Sestavení e-mailu
$subject = 'Poptávka – Horliko (' . $service . ')';

// From – některé hosty odmítají cizí domény, proto použijeme no-reply@vas-domeny
$host = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
$host = preg_replace('/[^a-zA-Z0-9.-]+/', '', $host);
$fromEmail = 'no-reply@' . ($host ?: 'localhost');
$fromName = 'Horliko podlahářství';

$headers = [];
$headers[] = 'MIME-Version: 1.0';
$headers[] = 'From: ' . mb_encode_mimeheader($fromName, 'UTF-8') . ' <' . $fromEmail . '>';
$headers[] = 'Reply-To: ' . $email;

if (count($attachments) > 0) {
  $boundary = '=_horliko_' . bin2hex(random_bytes(12));
  $headers[] = 'Content-Type: multipart/mixed; boundary="' . $boundary . '"';

  $body = "--{$boundary}\r\n";
  $body .= "Content-Type: text/html; charset=UTF-8\r\n";
  $body .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
  $body .= $htmlMessage . "\r\n";

  foreach ($attachments as $att) {
    $b64 = chunk_split(base64_encode($att['data']));
    $filename = $att['name'];
    $mime = $att['mime'];

    $body .= "--{$boundary}\r\n";
    $body .= "Content-Type: {$mime}; name=\"{$filename}\"\r\n";
    $body .= "Content-Transfer-Encoding: base64\r\n";
    $body .= "Content-Disposition: attachment; filename=\"{$filename}\"\r\n\r\n";
    $body .= $b64 . "\r\n";
  }

  $body .= "--{$boundary}--\r\n";
} else {
  $headers[] = 'Content-Type: text/html; charset=UTF-8';
  $body = $htmlMessage;
}

// Odeslání
$ok = @mail($TO_EMAIL, mb_encode_mimeheader($subject, 'UTF-8'), $body, implode("\r\n", $headers));

if (!$ok) {
  respond(false, 'Nepodařilo se odeslat e-mail. Zkuste to prosím později nebo nám zavolejte.', 500);
}

respond(true);
