<?php
// Horliko – contact form sender (HTML email + attachments)
// Returns JSON { ok: true } or { ok:false, error:"..." }

header('Content-Type: application/json; charset=utf-8');

function respond($ok, $msg = null, $http = 200) {
  http_response_code($http);
  echo json_encode($ok ? ["ok" => true] : ["ok" => false, "error" => $msg], JSON_UNESCAPED_UNICODE);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  respond(false, 'Metoda není povolena.', 405);
}

// basic fields
$name    = trim($_POST['name'] ?? '');
$email   = trim($_POST['email'] ?? '');
$phone   = trim($_POST['phone'] ?? '');
$topic   = trim($_POST['topic'] ?? '');
$message = trim($_POST['message'] ?? '');
$gdpr    = isset($_POST['gdpr']);

if ($name === '' || $email === '' || $topic === '' || $message === '' || !$gdpr) {
  respond(false, 'Chybí povinné údaje.');
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
  respond(false, 'Neplatný e-mail.');
}

// attachments validation
$allowedMime = [
  'image/jpeg' => 'jpg',
  'image/png'  => 'png',
  'application/pdf' => 'pdf'
];

$totalSize = 0;
$files = [];
if (isset($_FILES['attachment'])) {
  // support multiple
  $f = $_FILES['attachment'];
  $count = is_array($f['name']) ? count($f['name']) : 1;

  for ($i = 0; $i < $count; $i++) {
    $err  = is_array($f['error']) ? $f['error'][$i] : $f['error'];
    $tmp  = is_array($f['tmp_name']) ? $f['tmp_name'][$i] : $f['tmp_name'];
    $nameF= is_array($f['name']) ? $f['name'][$i] : $f['name'];
    $size = is_array($f['size']) ? (int)$f['size'][$i] : (int)$f['size'];

    if ($err === UPLOAD_ERR_NO_FILE) continue;
    if ($err !== UPLOAD_ERR_OK) {
      respond(false, 'Chyba při nahrávání přílohy.');
    }

    $totalSize += $size;
    if ($totalSize > 5 * 1024 * 1024) {
      respond(false, 'Celková velikost příloh je max 5 MB.');
    }

    $mime = mime_content_type($tmp);
    if (!isset($allowedMime[$mime])) {
      respond(false, 'Povolené přílohy: JPG, PNG, PDF.');
    }

    $safeName = preg_replace('/[^\w\-. ]+/u', '_', $nameF);
    $files[] = [
      'tmp' => $tmp,
      'name' => $safeName,
      'mime' => $mime,
    ];
  }
}

// recipients (change here if needed)
$to = 'hornik34@seznam.cz';

// subject
$subject = 'Nová poptávka – Horliko podlahářství';

// HTML body
$dt = date('d.m.Y H:i');
$esc = fn($s) => htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');

$html = '<html><body style="font-family:Arial,Helvetica,sans-serif;line-height:1.4;color:#111;">';
$html .= '<h2 style="margin:0 0 12px;">Nová poptávka z webu</h2>';
$html .= '<table cellpadding="8" cellspacing="0" style="border-collapse:collapse;border:1px solid #e5e5e5;">';
$html .= '<tr><td style="background:#f6f6f6;border:1px solid #e5e5e5;"><b>Datum</b></td><td style="border:1px solid #e5e5e5;">'.$esc($dt).'</td></tr>';
$html .= '<tr><td style="background:#f6f6f6;border:1px solid #e5e5e5;"><b>Jméno</b></td><td style="border:1px solid #e5e5e5;">'.$esc($name).'</td></tr>';
$html .= '<tr><td style="background:#f6f6f6;border:1px solid #e5e5e5;"><b>E-mail</b></td><td style="border:1px solid #e5e5e5;">'.$esc($email).'</td></tr>';
$html .= '<tr><td style="background:#f6f6f6;border:1px solid #e5e5e5;"><b>Telefon</b></td><td style="border:1px solid #e5e5e5;">'.$esc($phone ?: '—').'</td></tr>';
$html .= '<tr><td style="background:#f6f6f6;border:1px solid #e5e5e5;"><b>Služba</b></td><td style="border:1px solid #e5e5e5;">'.$esc($topic).'</td></tr>';
$html .= '</table>';
$html .= '<h3 style="margin:16px 0 8px;">Popis zakázky</h3>';
$html .= '<div style="white-space:pre-wrap;border:1px solid #e5e5e5;padding:10px;border-radius:6px;">'.$esc($message).'</div>';
$html .= '</body></html>';

$boundary = '=_horliko_' . md5(uniqid((string)mt_rand(), true));

$headers = [];
$headers[] = 'MIME-Version: 1.0';
$headers[] = 'From: Horliko podlahářství <no-reply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost') . '>';
$headers[] = 'Reply-To: ' . $email;

if (count($files) === 0) {
  $headers[] = 'Content-Type: text/html; charset=UTF-8';
  $ok = @mail($to, '=?UTF-8?B?'.base64_encode($subject).'?=', $html, implode("\r\n", $headers));
  if (!$ok) respond(false, 'E-mail se nepodařilo odeslat (mail() selhalo).', 500);
  respond(true);
}

// multipart with attachments
$headers[] = 'Content-Type: multipart/mixed; boundary="'.$boundary.'"';

$body  = "--$boundary\r\n";
$body .= "Content-Type: text/html; charset=UTF-8\r\n";
$body .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
$body .= $html . "\r\n";

foreach ($files as $file) {
  $content = @file_get_contents($file['tmp']);
  if ($content === false) respond(false, 'Nelze načíst přílohu.', 500);

  $body .= "--$boundary\r\n";
  $body .= "Content-Type: ".$file['mime']."; name=\"".$file['name']."\"\r\n";
  $body .= "Content-Transfer-Encoding: base64\r\n";
  $body .= "Content-Disposition: attachment; filename=\"".$file['name']."\"\r\n\r\n";
  $body .= chunk_split(base64_encode($content)) . "\r\n";
}
$body .= "--$boundary--\r\n";

$ok = @mail($to, '=?UTF-8?B?'.base64_encode($subject).'?=', $body, implode("\r\n", $headers));
if (!$ok) respond(false, 'E-mail se nepodařilo odeslat (mail() selhalo).', 500);

respond(true);
