<?php
require_once __DIR__ . '/db.php';

function h(?string $s): string {
    return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8');
}

function now_iso(): string {
    return gmdate('c');
}

function log_refresh(string $msg): void {
    @mkdir(__DIR__ . '/data', 0777, true);
    file_put_contents(REFRESH_LOG, '[' . date('Y-m-d H:i:s') . '] ' . $msg . PHP_EOL, FILE_APPEND);
}

function detect_shop(string $url): string {
    $host = parse_url($url, PHP_URL_HOST);
    $host = strtolower($host ?? '');

    $map = [
        'alza.cz' => 'alza',
        'www.alza.cz' => 'alza',
        'czc.cz' => 'czc',
        'www.czc.cz' => 'czc',
        'mall.cz' => 'mall',
        'www.mall.cz' => 'mall',
        'datart.cz' => 'datart',
        'www.datart.cz' => 'datart',
        'rohlik.cz' => 'rohlik',
        'www.rohlik.cz' => 'rohlik',
    ];
    if (isset($map[$host])) return $map[$host];

    $parts = array_values(array_filter(explode('.', $host)));
    if (count($parts) >= 2) return $parts[count($parts)-2];
    return $host ?: 'unknown';
}

function http_get(string $url): string {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => HTTP_TIMEOUT,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_USERAGENT => USER_AGENT,
        CURLOPT_HTTPHEADER => [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        ],
    ]);
    $out = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);

    if ($out === false || $code >= 400) {
        throw new RuntimeException('HTTP error: ' . ($err ?: 'status ' . $code));
    }
    return (string)$out;
}

function to_float_price(?string $text): ?float {
    if (!$text) return null;
    $t = preg_replace('/[^0-9,\.]/u', '', $text);
    $t = str_replace(' ', '', $t ?? '');
    if ($t === '') return null;

    if (substr_count($t, ',') === 1 && substr_count($t, '.') === 0) {
        $t = str_replace(',', '.', $t);
    }
    if (substr_count($t, '.') >= 1 && substr_count($t, ',') === 1) {
        $t = str_replace('.', '', $t);
        $t = str_replace(',', '.', $t);
    }
    if (!is_numeric($t)) return null;
    return (float)$t;
}

function parse_og(DOMXPath $xp): array {
    $title = null;
    $img = null;

    $t = $xp->query("//meta[@property='og:title']");
    if ($t && $t->length && $t->item(0)->getAttribute('content')) {
        $title = trim($t->item(0)->getAttribute('content'));
    }

    $i = $xp->query("//meta[@property='og:image']");
    if ($i && $i->length && $i->item(0)->getAttribute('content')) {
        $img = trim($i->item(0)->getAttribute('content'));
    }

    if (!$title) {
        $tt = $xp->query('//title');
        if ($tt && $tt->length) $title = trim($tt->item(0)->textContent);
    }

    return [$title, $img];
}

function extract_jsonld_product(string $html): array {
    $matches = [];
    preg_match_all('/<script[^>]+type=["\']application\/ld\+json["\'][^>]*>(.*?)<\/script>/si', $html, $matches);
    if (empty($matches[1])) return [null, null, null];

    foreach ($matches[1] as $raw) {
        $raw = trim($raw);
        if ($raw === '') continue;

        $data = json_decode($raw, true);
        if (!is_array($data)) continue;

        $candidates = $data;
        if (array_keys($data) !== range(0, count($data)-1)) {
            $candidates = [$data];
        }

        $expanded = [];
        foreach ($candidates as $item) {
            if (is_array($item) && isset($item['@graph']) && is_array($item['@graph'])) {
                foreach ($item['@graph'] as $g) if (is_array($g)) $expanded[] = $g;
            } elseif (is_array($item)) {
                $expanded[] = $item;
            }
        }

        foreach ($expanded as $item) {
            $type = $item['@type'] ?? null;
            $isProduct = false;
            if (is_string($type) && strtolower($type) === 'product') $isProduct = true;
            if (is_array($type)) foreach ($type as $t) if (is_string($t) && strtolower($t) === 'product') $isProduct = true;
            if (!$isProduct) continue;

            $name = is_string($item['name'] ?? null) ? $item['name'] : null;

            $offers = $item['offers'] ?? null;
            $price = null;
            $currency = null;

            $parseOffer = function($off) use (&$price, &$currency) {
                if (!is_array($off)) return;
                if ($price === null) {
                    $p = $off['price'] ?? ($off['lowPrice'] ?? ($off['highPrice'] ?? null));
                    if (is_numeric($p)) $price = (float)$p;
                    if (is_string($p)) $price = to_float_price($p);
                }
                if ($currency === null && is_string($off['priceCurrency'] ?? null)) {
                    $currency = $off['priceCurrency'];
                }
            };

            if (is_array($offers) && array_keys($offers) !== range(0, count($offers)-1)) {
                $parseOffer($offers);
            } elseif (is_array($offers)) {
                foreach ($offers as $off) $parseOffer($off);
            }

            if ($price !== null) return [$name, $price, $currency];
        }
    }

    return [null, null, null];
}

function scrape_product(string $url): array {
    $html = http_get($url);

    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xp = new DOMXPath($doc);

    [$name_ld, $price_ld, $cur_ld] = extract_jsonld_product($html);
    [$title_og, $img_og] = parse_og($xp);

    $name = $name_ld ?: $title_og;
    $price = $price_ld;
    $currency = strtoupper($cur_ld ?: 'CZK');

    if ($price === null) {
        $m = $xp->query("//meta[@itemprop='price']");
        if ($m && $m->length) {
            $content = $m->item(0)->getAttribute('content');
            $price = to_float_price($content);
        }
    }

    if ($price === null) {
        if (preg_match('/(\d[\d\s\.,]{1,12})\s*Kč/u', strip_tags($html), $m)) {
            $price = to_float_price($m[1]);
            $currency = 'CZK';
        }
    }

    return [
        'name' => $name,
        'price' => $price,
        'currency' => $currency,
        'image_url' => $img_og,
    ];
}

function download_image(?string $image_url, string $filename_hint): ?string {
    if (!$image_url) return null;

    @mkdir(IMAGES_DIR, 0777, true);

    $path = parse_url($image_url, PHP_URL_PATH) ?? '';
    $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION) ?: '');
    if (!in_array($ext, ['jpg','jpeg','png','webp'])) $ext = 'jpg';

    $safe = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename_hint);
    $safe = substr($safe ?: 'product', 0, 80);

    $rel = 'images/' . $safe . '.' . $ext;
    $abs = STATIC_DIR . '/' . $rel;

    $ch = curl_init($image_url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => HTTP_TIMEOUT,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_USERAGENT => USER_AGENT,
    ]);
    $bin = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);

    if ($bin === false || $code >= 400) return null;
    file_put_contents($abs, $bin);
    return $rel;
}

function add_price_history(PDO $db, int $product_id, float $price, string $currency): void {
    $st = $db->prepare("INSERT INTO price_history(product_id, price, currency, checked_at) VALUES(?,?,?,?)");
    $st->execute([$product_id, $price, $currency, now_iso()]);
}

function notify_drop(array $product, float $old, float $new): void {
    $subject = 'Cena klesla: ' . (($product['name'] ?? '') ?: ($product['shop'] ?? 'produkt'));
    $body = "Produkt: " . (($product['name'] ?? '') ?: '-') . "\n"
          . "Obchod: " . ($product['shop'] ?? '-') . "\n"
          . "Stará cena: " . number_format($old, 0, ',', ' ') . " Kč\n"
          . "Nová cena: " . number_format($new, 0, ',', ' ') . " Kč\n"
          . "Odkaz: " . ($product['url'] ?? '-') . "\n";

    if (NOTIFY_EMAIL) {
        @mail(NOTIFY_EMAIL, $subject, $body);
    }
    log_refresh("NOTIFY: " . $subject . " | " . str_replace("\n", " | ", $body));
}
