function $(id){ return document.getElementById(id); }

const state = { tags: [], selected: new Set(), photos: [] };

function qsFromSelected() {
  const tags = Array.from(state.selected);
  return tags.length ? `&tags=${encodeURIComponent(tags.join(","))}` : "";
}

async function loadTags() {
  const res = await fetch(`api.php?action=tags`);
  const data = await res.json();
  state.tags = data.tags || [];
}

async function loadPhotos() {
  const res = await fetch(`api.php?action=photos${qsFromSelected()}`);
  const data = await res.json();
  state.photos = data.photos || [];
}

function renderChips(chipsEl) {
  chipsEl.innerHTML = "";
  for (const t of state.tags) {
    const b = document.createElement("button");
    b.className = "chip";
    b.type = "button";
    b.textContent = `#${t}`;
    b.setAttribute("aria-pressed", state.selected.has(t) ? "true" : "false");
    b.addEventListener("click", async () => {
      state.selected.has(t) ? state.selected.delete(t) : state.selected.add(t);
      renderChips(chipsEl);
      await loadPhotos();
      renderGrid($("grid"));
    });
    chipsEl.appendChild(b);
  }
}

function renderGrid(gridEl) {
  if (!gridEl) return;
  gridEl.innerHTML = "";
  if (!state.photos.length) {
    const p = document.createElement("p");
    p.className = "sub";
    p.textContent = "Nic nenalezeno.";
    gridEl.appendChild(p);
    return;
  }

  for (const ph of state.photos) {
    const card = document.createElement("div");
    card.className = "card";
    card.tabIndex = 0;

    const img = document.createElement("img");
    img.className = "thumb";
    img.src = `uploads/${ph.filename}`;
    img.alt = ph.title;

    const meta = document.createElement("div");
    meta.className = "meta";

    const title = document.createElement("div");
    title.className = "title";
    title.textContent = ph.title;

    const tagsline = document.createElement("div");
    tagsline.className = "tagsline";
    for (const t of (ph.tags || [])) {
      const s = document.createElement("span");
      s.className = "badge";
      s.textContent = `#${t}`;
      tagsline.appendChild(s);
    }

    meta.appendChild(title);
    meta.appendChild(tagsline);

    card.appendChild(img);
    card.appendChild(meta);

    const open = () => openLightbox(ph);
    card.addEventListener("click", open);
    card.addEventListener("keydown", (e) => {
      if (e.key === "Enter" || e.key === " ") open();
    });

    gridEl.appendChild(card);
  }
}

function openLightbox(ph) {
  const lightbox = $("lightbox");
  const lbImg = $("lbImg");
  const lbTitle = $("lbTitle");
  const lbMeta = $("lbMeta");
  const downloadBtn = $("downloadBtn");
  if (!lightbox || !lbImg || !lbTitle || !lbMeta || !downloadBtn) return;

  const url = `uploads/${ph.filename}`;
  lbImg.src = url;
  lbImg.alt = ph.title;
  lbTitle.textContent = ph.title;
  lbMeta.textContent = (ph.tags || []).map(t => `#${t}`).join("  ");
  downloadBtn.href = url;
  downloadBtn.setAttribute("download", ph.filename);
  lightbox.classList.add("open");
}

function closeLightbox() {
  const lightbox = $("lightbox");
  const lbImg = $("lbImg");
  if (!lightbox) return;
  lightbox.classList.remove("open");
  if (lbImg) lbImg.src = "";
}

async function init() {
  const chipsEl = $("chips");
  const gridEl = $("grid");
  const clearBtn = $("clearBtn");
  const closeBtn = $("closeBtn");
  const lightbox = $("lightbox");

  const missing = [];
  if (!chipsEl) missing.push("#chips");
  if (!gridEl) missing.push("#grid");
  if (!clearBtn) missing.push("#clearBtn");
  if (!closeBtn) missing.push("#closeBtn");
  if (!lightbox) missing.push("#lightbox");
  if (missing.length) {
    console.error("Chybí prvky v HTML:", missing.join(", "));
    return;
  }

  clearBtn.addEventListener("click", async () => {
    state.selected.clear();
    renderChips(chipsEl);
    await loadPhotos();
    renderGrid(gridEl);
  });

  closeBtn.addEventListener("click", closeLightbox);
  lightbox.addEventListener("click", (e) => { if (e.target === lightbox) closeLightbox(); });
  document.addEventListener("keydown", (e) => { if (e.key === "Escape") closeLightbox(); });

  await loadTags();
  await loadPhotos();
  renderChips(chipsEl);
  renderGrid(gridEl);
}

window.addEventListener("DOMContentLoaded", () => {
  init().catch(err => console.error("Init error:", err));
});
