document.addEventListener('DOMContentLoaded', function () {
    // Smooth scroll for anchor links
    const links = document.querySelectorAll('a[href^="#"]');
    links.forEach(link => {
        link.addEventListener('click', function (e) {
            const targetId = this.getAttribute('href').substring(1);
            const target = document.getElementById(targetId);
            if (target) {
                e.preventDefault();
                window.scrollTo({
                    top: target.offsetTop - 80,
                    behavior: 'smooth'
                });
            }
        });
    });

    // Navbar shadow on scroll
    const navbar = document.querySelector('.navbar');
    if (navbar) {
        window.addEventListener('scroll', () => {
            if (window.scrollY > 10) {
                navbar.classList.add('navbar-scrolled');
            } else {
                navbar.classList.remove('navbar-scrolled');
            }
        });
    }

    // Back to top button
    const backToTop = document.querySelector('.back-to-top');
    if (backToTop) {
        window.addEventListener('scroll', () => {
            if (window.scrollY > 400) {
                backToTop.classList.add('visible');
            } else {
                backToTop.classList.remove('visible');
            }
        });
        backToTop.addEventListener('click', (e) => {
            e.preventDefault();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });
    }
});

// Galaxy intro scroll animation
const introSection = document.querySelector('.galaxy-intro');
if (introSection) {
    const viewport = introSection.querySelector('.galaxy-viewport');
    const layers = {
        space: introSection.querySelector('.layer-space'),
        planet: introSection.querySelector('.layer-planet'),
        continent: introSection.querySelector('.layer-continent'),
        city: introSection.querySelector('.layer-city'),
        district: introSection.querySelector('.layer-district'),
    };

    const LAYER_KEYS = ['space', 'planet', 'continent', 'city', 'district'];

    function updateGalaxyAnimation() {
        const rect = introSection.getBoundingClientRect();
        const windowHeight = window.innerHeight || document.documentElement.clientHeight;

        // When section is not visible at all, skip heavy work
        if (rect.bottom < 0 || rect.top > windowHeight) {
            return;
        }

        // progress from 0 to 1 across the intro section
        const totalScrollable = introSection.offsetHeight - windowHeight;
        let rawProgress = 0;
        if (totalScrollable > 0) {
            const scrolledFromTop = window.scrollY - introSection.offsetTop;
            rawProgress = scrolledFromTop / totalScrollable;
        }
        const progress = Math.min(1, Math.max(0, rawProgress));

        // Each layer gets a slice of the progress range
        const segment = 1 / (LAYER_KEYS.length - 1); // last two overlap slightly
        LAYER_KEYS.forEach((key, index) => {
            const layer = layers[key];
            if (!layer) return;

            // Start and end for this layer
            const start = Math.max(0, (index - 1) * segment);
            const end = Math.min(1, index * segment + segment * 0.7);

            let opacity = 0;
            if (progress >= start && progress <= end) {
                const local = (progress - start) / (end - start);
                opacity = Math.max(0, Math.min(1, local));
            }
            layer.style.opacity = opacity.toFixed(3);

            // Slight parallax / zoom based on progress
            const scaleBase = 1.05;
            const extraScale = 0.08 * opacity;
            const translateY = (0.5 - progress) * 10; // subtle vertical shift
            layer.style.transform = `scale(${(scaleBase + extraScale).toFixed(3)}) translateY(${translateY.toFixed(1)}px)`;
        });

        // Fade out overlay and hint near the end of the intro,
        // so by the time we reach the hero section, they don't overlap
        const overlay = introSection.querySelector('.galaxy-overlay');
        const hint = introSection.querySelector('.scroll-hint');
        const fadeStart = 0.65;
        const fadeEnd = 0.95;
        const fadeRange = fadeEnd - fadeStart;

        let overlayOpacity = 1;
        if (progress >= fadeStart) {
            const local = Math.min(1, Math.max(0, (progress - fadeStart) / fadeRange));
            overlayOpacity = 1 - local;
        }
        if (overlay) overlay.style.opacity = overlayOpacity.toFixed(3);
        if (hint) hint.style.opacity = overlayOpacity.toFixed(3);
    }

    // Initial state
    updateGalaxyAnimation();
    window.addEventListener('scroll', updateGalaxyAnimation);
    window.addEventListener('resize', updateGalaxyAnimation);
}
