\
<?php
declare(strict_types=1);

require_once __DIR__ . '/../app/repo.php';
require_once __DIR__ . '/partials.php';
require_once __DIR__ . '/../app/util.php';

$page = $_GET['page'] ?? 'dashboard';
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

function redirect(string $page, array $extra = []): void {
    $qs = http_build_query(array_merge(['page'=>$page], $extra));
    header("Location: /?$qs");
    exit;
}

function post_str(string $k, string $default=''): string {
    return isset($_POST[$k]) ? trim((string)$_POST[$k]) : $default;
}
function post_float(string $k, float $default=0.0): float {
    $v = post_str($k, '');
    $v = str_replace([' ', ','], ['', '.'], $v);
    return is_numeric($v) ? (float)$v : $default;
}
function post_int_or_null(string $k): ?int {
    if (!isset($_POST[$k]) || $_POST[$k]==='') return null;
    return (int)$_POST[$k];
}

$flash = $_GET['ok'] ?? null;

/* -------------------- Actions (POST) -------------------- */
if ($method === 'POST') {
    $action = $_POST['action'] ?? '';
    try {
        if ($action === 'add_income') {
            $source = post_str('source','salary');
            add_transaction([
                'date' => parse_date(post_str('date'), today()),
                'type' => 'income',
                'source' => $source,
                'amount' => post_float('amount', 0.0),
                'merchant' => post_str('merchant',''),
                'note' => post_str('note',''),
                'category_id' => null,
            ]);
            redirect('income', ['ok'=>'Příjem uložen.']);
        }

        if ($action === 'add_expense') {
            add_transaction([
                'date' => parse_date(post_str('date'), today()),
                'type' => 'expense',
                'source' => null,
                'amount' => post_float('amount', 0.0),
                'merchant' => post_str('merchant',''),
                'note' => post_str('note',''),
                'category_id' => post_int_or_null('category_id'),
            ]);
            redirect('expenses', ['ok'=>'Výdaj uložen.']);
        }

        if ($action === 'add_category') {
            $name = post_str('name','');
            if ($name !== '') add_category($name);
            redirect('categories', ['ok'=>'Kategorie uložena.']);
        }

        if ($action === 'add_loan') {
            add_loan([
                'name' => post_str('name',''),
                'principal_total' => post_float('principal_total', 0.0),
                'start_date' => parse_date(post_str('start_date'), today()),
                'planned_monthly_payment' => (post_str('planned_monthly_payment','')==='') ? null : post_float('planned_monthly_payment', 0.0),
                'note' => post_str('note',''),
            ]);
            redirect('loans', ['ok'=>'Úvěr uložen.']);
        }

        if ($action === 'add_payment') {
            add_loan_payment([
                'loan_id' => (int)post_str('loan_id','0'),
                'date' => parse_date(post_str('date'), today()),
                'amount' => post_float('amount', 0.0),
                'note' => post_str('note',''),
            ]);
            redirect('loan_detail', ['id'=>(int)post_str('loan_id','0'), 'ok'=>'Splátka uložena.']);
        }

    } catch (Throwable $e) {
        // Basic safe error output in UI
        redirect($page, ['ok'=>'Chyba: '.$e->getMessage()]);
    }
}

/* -------------------- Actions (GET delete) -------------------- */
if (isset($_GET['del_tx'])) {
    delete_transaction((int)$_GET['del_tx']);
    redirect($_GET['back'] ?? 'dashboard', ['ok'=>'Smazáno.']);
}
if (isset($_GET['del_cat'])) {
    delete_category((int)$_GET['del_cat']);
    redirect('categories', ['ok'=>'Kategorie smazána.']);
}
if (isset($_GET['del_loan'])) {
    delete_loan((int)$_GET['del_loan']);
    redirect('loans', ['ok'=>'Úvěr smazán.']);
}
if (isset($_GET['del_pay'])) {
    delete_loan_payment((int)$_GET['del_pay']);
    redirect('loan_detail', ['id'=>(int)($_GET['id'] ?? 0), 'ok'=>'Splátka smazána.']);
}

/* -------------------- Pages -------------------- */
if ($page === 'dashboard') {
    render_header('Přehled', 'dashboard');

    // Filters
    $period = $_GET['period'] ?? 'this_month'; // this_month | last_month | year | custom
    $type = $_GET['type'] ?? '';
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    $year = $_GET['year'] ?? '';
    $month = $_GET['month'] ?? '';
    $category_ids = $_GET['category_ids'] ?? [];
    if (!is_array($category_ids)) $category_ids = [$category_ids];

    // calculate date range
    $df = null; $dt = null;
    $now = new DateTimeImmutable('now');
    if ($period === 'this_month') {
        [$df,$dt] = month_range($now->format('Y-m'));
    } elseif ($period === 'last_month') {
        $lm = $now->modify('first day of last month')->format('Y-m');
        [$df,$dt] = month_range($lm);
    } elseif ($period === 'year') {
        [$df,$dt] = year_range($year ?: $now->format('Y'));
    } else { // custom
        $df = $date_from ? parse_date($date_from, $now->format('Y-m-01')) : null;
        $dt = $date_to ? parse_date($date_to, $now->modify('+1 day')->format('Y-m-d')) : null;
        if ($dt) {
            // end is exclusive; add +1 day for inclusive UI input
            $dtObj = DateTimeImmutable::createFromFormat('Y-m-d', $dt);
            if ($dtObj) $dt = $dtObj->modify('+1 day')->format('Y-m-d');
        }
    }

    $filter = ['date_from'=>$df, 'date_to'=>$dt];
    $kpis = totals($filter);

    // loans summary
    $loans = list_loans();
    $loan_remaining_sum = 0.0;
    foreach ($loans as $l) $loan_remaining_sum += (float)$l['remaining'];

    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';

    ?>
    <h1>Přehled</h1>

    <div class="card">
      <form method="get" action="/">
        <input type="hidden" name="page" value="dashboard">
        <div class="row">
          <label>Období
            <select name="period">
              <option value="this_month" <?= $period==='this_month'?'selected':'' ?>>Tento měsíc</option>
              <option value="last_month" <?= $period==='last_month'?'selected':'' ?>>Minulý měsíc</option>
              <option value="year" <?= $period==='year'?'selected':'' ?>>Rok</option>
              <option value="custom" <?= $period==='custom'?'selected':'' ?>>Vlastní</option>
            </select>
          </label>
          <label>Rok (pro „Rok“)
            <input name="year" value="<?= h($year ?: $now->format('Y')) ?>" placeholder="2026">
          </label>
          <label>Typ
            <select name="type">
              <option value="" <?= $type===''?'selected':'' ?>>Vše</option>
              <option value="income" <?= $type==='income'?'selected':'' ?>>Příjmy</option>
              <option value="expense" <?= $type==='expense'?'selected':'' ?>>Výdaje</option>
            </select>
          </label>
          <label>Vlastní od (pro „Vlastní“)
            <input type="date" name="date_from" value="<?= h($date_from) ?>">
          </label>
          <label>Vlastní do (pro „Vlastní“)
            <input type="date" name="date_to" value="<?= h($date_to) ?>">
          </label>
        </div>

        <label>Kategorie výdajů (volitelné)
          <?php $cats = get_categories(); ?>
          <select id="category_ids" name="category_ids[]" multiple size="6">
            <?php foreach ($cats as $c): ?>
              <option value="<?= (int)$c['id'] ?>" <?= in_array((string)$c['id'], array_map('strval',$category_ids), true) ? 'selected':'' ?>>
                <?= h($c['name']) ?>
              </option>
            <?php endforeach; ?>
          </select>
          <div class="small">Vybráno: <span id="category_preview">—</span></div>
        </label>

        <button type="submit">Použít filtr</button>
      </form>
    </div>

    <br>

    <div class="grid">
      <div class="card kpi">
        <div class="label">Celkové příjmy (období)</div>
        <div class="value"><?= money((float)$kpis['income']) ?></div>
      </div>
      <div class="card kpi">
        <div class="label">Celkové výdaje (období)</div>
        <div class="value"><?= money((float)$kpis['expense']) ?></div>
      </div>
      <div class="card kpi">
        <div class="label">Bilance (období)</div>
        <?php $bal = (float)$kpis['balance']; ?>
        <div class="value <?= $bal>=0?'good':'bad' ?>"><?= money($bal) ?></div>
      </div>
      <div class="card kpi">
        <div class="label">Úvěry: zbývá doplatit</div>
        <div class="value"><?= money($loan_remaining_sum) ?></div>
      </div>
    </div>

    <br>

    <?php
      $txFilter = ['date_from'=>$df,'date_to'=>$dt,'type'=>$type?:null,'category_ids'=>$category_ids];
      $tx = list_transactions($txFilter);
      $break = expense_breakdown_by_category(['date_from'=>$df,'date_to'=>$dt,'category_ids'=>$category_ids]);
    ?>

    <div class="row">
      <div class="card">
        <h2>Výdaje podle kategorií</h2>
        <table>
          <thead><tr><th>Kategorie</th><th>Celkem</th></tr></thead>
          <tbody>
          <?php foreach ($break as $b): ?>
            <tr><td><?= h($b['category']) ?></td><td><?= money((float)$b['total']) ?></td></tr>
          <?php endforeach; ?>
          <?php if (!$break): ?>
            <tr><td colspan="2" class="small">Žádná data.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
      <div class="card">
        <h2>Úvěry</h2>
        <table>
          <thead><tr><th>Název</th><th>Zbývá</th><th>Detail</th></tr></thead>
          <tbody>
          <?php foreach ($loans as $l): ?>
            <tr>
              <td><?= h($l['name']) ?></td>
              <td><?= money((float)$l['remaining']) ?></td>
              <td><a href="/?page=loan_detail&id=<?= (int)$l['id'] ?>">Otevřít</a></td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$loans): ?>
            <tr><td colspan="3" class="small">Zatím žádné úvěry.</td></tr>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>

    <br>

    <div class="card">
      <h2>Transakce (max 500)</h2>
      <table>
        <thead>
          <tr>
            <th>Datum</th><th>Typ</th><th>Částka</th><th>Kategorie</th><th>Popisek</th><th></th>
          </tr>
        </thead>
        <tbody>
        <?php foreach ($tx as $t): ?>
          <tr>
            <td><?= h($t['date']) ?></td>
            <td><span class="pill"><?= h($t['type']) ?><?= $t['type']==='income' && $t['source'] ? ' · '.h($t['source']) : '' ?></span></td>
            <td class="<?= $t['type']==='income'?'good':'bad' ?>"><?= money((float)$t['amount']) ?></td>
            <td><?= h($t['category_name'] ?? '') ?></td>
            <td>
              <div><?= h($t['merchant'] ?? '') ?></div>
              <?php if (!empty($t['note'])): ?><div class="small"><?= h($t['note']) ?></div><?php endif; ?>
            </td>
            <td>
              <a class="danger" data-confirm="Smazat transakci?" href="/?del_tx=<?= (int)$t['id'] ?>&back=dashboard">Smazat</a>
            </td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$tx): ?>
          <tr><td colspan="6" class="small">Žádná data.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>

<?php
    render_footer();
    exit;
}

/* Income page */
if ($page === 'income') {
    render_header('Admin – Příjmy', 'income');
    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';

    $tx = list_transactions(['type'=>'income','date_from'=>null,'date_to'=>null,'category_ids'=>[]]);
?>
  <h1>Admin: Příjmy</h1>
  <div class="row">
    <div class="card">
      <h2>Přidat příjem</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_income">
        <label>Datum <input type="date" name="date" value="<?= h(today()) ?>"></label>
        <label>Typ příjmu
          <select name="source">
            <option value="salary">Plat</option>
            <option value="other">Další příjem</option>
          </select>
        </label>
        <label>Částka <input name="amount" placeholder="např. 35000"></label>
        <label>Popisek (např. firma / zakázka) <input name="merchant"></label>
        <label>Poznámka <textarea name="note"></textarea></label>
        <button>Uložit</button>
      </form>
    </div>
    <div class="card">
      <h2>Poslední příjmy</h2>
      <table>
        <thead><tr><th>Datum</th><th>Částka</th><th>Zdroj</th><th>Popisek</th><th></th></tr></thead>
        <tbody>
        <?php foreach ($tx as $t): ?>
          <tr>
            <td><?= h($t['date']) ?></td>
            <td class="good"><?= money((float)$t['amount']) ?></td>
            <td><?= h($t['source'] ?? '') ?></td>
            <td><?= h($t['merchant'] ?? '') ?><div class="small"><?= h($t['note'] ?? '') ?></div></td>
            <td><a class="danger" data-confirm="Smazat?" href="/?del_tx=<?= (int)$t['id'] ?>&back=income">Smazat</a></td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$tx): ?><tr><td colspan="5" class="small">Zatím nic.</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
<?php
    render_footer();
    exit;
}

/* Expenses page */
if ($page === 'expenses') {
    render_header('Admin – Výdaje', 'expenses');
    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';
    $cats = get_categories();
    $tx = list_transactions(['type'=>'expense','date_from'=>null,'date_to'=>null,'category_ids'=>[]]);
?>
  <h1>Admin: Výdaje</h1>
  <div class="row">
    <div class="card">
      <h2>Přidat výdaj</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_expense">
        <label>Datum <input type="date" name="date" value="<?= h(today()) ?>"></label>
        <label>Kategorie
          <select name="category_id">
            <option value="">(Bez kategorie)</option>
            <?php foreach ($cats as $c): ?>
              <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </label>
        <label>Částka <input name="amount" placeholder="např. 450"></label>
        <label>Obchod / popisek <input name="merchant" placeholder="např. Lidl, Nájem, Benzín"></label>
        <label>Poznámka <textarea name="note"></textarea></label>
        <button>Uložit</button>
      </form>
      <div class="small">Tip: Kategorie spravuješ v záložce „Kategorie“.</div>
    </div>
    <div class="card">
      <h2>Poslední výdaje</h2>
      <table>
        <thead><tr><th>Datum</th><th>Částka</th><th>Kategorie</th><th>Popisek</th><th></th></tr></thead>
        <tbody>
        <?php foreach ($tx as $t): ?>
          <tr>
            <td><?= h($t['date']) ?></td>
            <td class="bad"><?= money((float)$t['amount']) ?></td>
            <td><?= h($t['category_name'] ?? '') ?></td>
            <td><?= h($t['merchant'] ?? '') ?><div class="small"><?= h($t['note'] ?? '') ?></div></td>
            <td><a class="danger" data-confirm="Smazat?" href="/?del_tx=<?= (int)$t['id'] ?>&back=expenses">Smazat</a></td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$tx): ?><tr><td colspan="5" class="small">Zatím nic.</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
<?php
    render_footer();
    exit;
}

/* Categories page */
if ($page === 'categories') {
    render_header('Kategorie', 'categories');
    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';
    $cats = get_categories();
?>
  <h1>Kategorie výdajů</h1>
  <div class="row">
    <div class="card">
      <h2>Přidat kategorii</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_category">
        <label>Název <input name="name" placeholder="např. Jídlo"></label>
        <button>Uložit</button>
      </form>
      <hr>
      <div class="small">Mazání kategorie nesmaže výdaje — jen se u nich nastaví „Bez kategorie“.</div>
    </div>
    <div class="card">
      <h2>Seznam</h2>
      <table>
        <thead><tr><th>Název</th><th></th></tr></thead>
        <tbody>
        <?php foreach ($cats as $c): ?>
          <tr>
            <td><?= h($c['name']) ?></td>
            <td><a class="danger" data-confirm="Smazat kategorii?" href="/?del_cat=<?= (int)$c['id'] ?>">Smazat</a></td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$cats): ?><tr><td colspan="2" class="small">Zatím žádné kategorie.</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
<?php
    render_footer();
    exit;
}

/* Loans list */
if ($page === 'loans') {
    render_header('Úvěry', 'loans');
    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';
    $loans = list_loans();
?>
  <h1>Úvěry</h1>
  <div class="row">
    <div class="card">
      <h2>Přidat úvěr</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_loan">
        <label>Název <input name="name" placeholder="např. Auto"></label>
        <label>Celková suma úvěru <input name="principal_total" placeholder="např. 250000"></label>
        <label>Datum začátku <input type="date" name="start_date" value="<?= h(today()) ?>"></label>
        <label>Plánovaná měsíční splátka (volitelné) <input name="planned_monthly_payment" placeholder="např. 5500"></label>
        <label>Poznámka <textarea name="note"></textarea></label>
        <button>Uložit</button>
      </form>
    </div>
    <div class="card">
      <h2>Seznam úvěrů</h2>
      <table>
        <thead><tr><th>Název</th><th>Celkem</th><th>Zbývá</th><th></th><th></th></tr></thead>
        <tbody>
        <?php foreach ($loans as $l): ?>
          <tr>
            <td><?= h($l['name']) ?></td>
            <td><?= money((float)$l['principal_total']) ?></td>
            <td><?= money((float)$l['remaining']) ?></td>
            <td><a href="/?page=loan_detail&id=<?= (int)$l['id'] ?>">Detail</a></td>
            <td><a class="danger" data-confirm="Smazat úvěr i splátky?" href="/?del_loan=<?= (int)$l['id'] ?>">Smazat</a></td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$loans): ?><tr><td colspan="5" class="small">Zatím žádné úvěry.</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
<?php
    render_footer();
    exit;
}

/* Loan detail */
if ($page === 'loan_detail') {
    $id = (int)($_GET['id'] ?? 0);
    $loan = $id ? get_loan($id) : null;
    if (!$loan) {
        redirect('loans', ['ok'=>'Úvěr nenalezen.']);
    }
    render_header('Detail úvěru', 'loans');
    if ($flash) echo '<div class="notice">'.h($flash).'</div><br>';

    $remaining = (float)$loan['remaining'];
    $payoff = estimate_payoff_date($remaining, $loan['planned_monthly_payment'] !== null ? (float)$loan['planned_monthly_payment'] : null, $loan['payments']);
?>
  <h1>Úvěr: <?= h($loan['name']) ?></h1>

  <div class="grid">
    <div class="card kpi">
      <div class="label">Celková suma</div>
      <div class="value"><?= money((float)$loan['principal_total']) ?></div>
    </div>
    <div class="card kpi">
      <div class="label">Splaceno</div>
      <div class="value good"><?= money((float)$loan['paid']) ?></div>
    </div>
    <div class="card kpi">
      <div class="label">Zbývá doplatit</div>
      <div class="value <?= $remaining<=0?'good':'bad' ?>"><?= money($remaining) ?></div>
    </div>
    <div class="card kpi">
      <div class="label">Odhad splacení</div>
      <div class="value"><?= $payoff ? h($payoff) : '—' ?></div>
      <div class="small">Odhad podle plánované splátky, nebo průměru posledních splátek.</div>
    </div>
  </div>

  <br>

  <div class="row">
    <div class="card">
      <h2>Přidat splátku</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_payment">
        <input type="hidden" name="loan_id" value="<?= (int)$loan['id'] ?>">
        <label>Datum <input type="date" name="date" value="<?= h(today()) ?>"></label>
        <label>Částka <input name="amount" placeholder="např. 5500"></label>
        <label>Poznámka <textarea name="note"></textarea></label>
        <button>Uložit</button>
      </form>
    </div>

    <div class="card">
      <h2>Splátky</h2>
      <table>
        <thead><tr><th>Datum</th><th>Částka</th><th>Poznámka</th><th></th></tr></thead>
        <tbody>
        <?php foreach ($loan['payments'] as $p): ?>
          <tr>
            <td><?= h($p['date']) ?></td>
            <td><?= money((float)$p['amount']) ?></td>
            <td class="small"><?= h($p['note'] ?? '') ?></td>
            <td><a class="danger" data-confirm="Smazat splátku?" href="/?page=loan_detail&id=<?= (int)$loan['id'] ?>&del_pay=<?= (int)$p['id'] ?>">Smazat</a></td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$loan['payments']): ?><tr><td colspan="4" class="small">Zatím žádné splátky.</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

<?php
    render_footer();
    exit;
}

/* fallback */
redirect('dashboard');
