\
<?php
declare(strict_types=1);

function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

function money(float $v): string {
    // Czech style formatting: 1 234,56
    return number_format($v, 2, ',', ' ') . ' Kč';
}

function today(): string { return (new DateTimeImmutable('now'))->format('Y-m-d'); }

function parse_date(?string $s, string $fallback): string {
    if (!$s) return $fallback;
    $d = DateTimeImmutable::createFromFormat('Y-m-d', $s);
    return $d ? $d->format('Y-m-d') : $fallback;
}

function month_range(?string $ym): array {
    // $ym: YYYY-MM or null
    $now = new DateTimeImmutable('now');
    if ($ym && preg_match('/^\d{4}-\d{2}$/', $ym)) {
        $start = DateTimeImmutable::createFromFormat('Y-m', $ym);
        if (!$start) $start = $now->modify('first day of this month');
    } else {
        $start = $now->modify('first day of this month');
    }
    $start = $start->setTime(0,0,0);
    $end = $start->modify('first day of next month');
    return [$start->format('Y-m-d'), $end->format('Y-m-d')];
}

function year_range(?string $y): array {
    $now = new DateTimeImmutable('now');
    if ($y && preg_match('/^\d{4}$/', $y)) {
        $start = DateTimeImmutable::createFromFormat('Y-m-d', $y.'-01-01');
        if (!$start) $start = $now->setDate((int)$now->format('Y'),1,1);
    } else {
        $start = $now->setDate((int)$now->format('Y'),1,1);
    }
    $end = $start->modify('+1 year');
    return [$start->format('Y-m-d'), $end->format('Y-m-d')];
}

function estimate_payoff_date(float $remaining, ?float $plannedMonthly, array $payments): ?string {
    // payments: array of ['date'=>..., 'amount'=>...], ordered desc or asc; we'll compute average monthly from last 3 payments.
    if ($remaining <= 0) return today();
    $monthly = null;
    if ($plannedMonthly !== null && $plannedMonthly > 0) {
        $monthly = $plannedMonthly;
    } else {
        // average of last up to 3 payments, normalized per month (assume monthly cadence)
        $n = min(3, count($payments));
        if ($n > 0) {
            $sum = 0.0;
            for ($i=0; $i<$n; $i++) $sum += (float)$payments[$i]['amount'];
            $monthly = $sum / $n;
        }
    }
    if (!$monthly || $monthly <= 0) return null;
    $months = (int)ceil($remaining / $monthly);
    $dt = new DateTimeImmutable('now');
    $payoff = $dt->modify('first day of this month')->modify("+$months month");
    return $payoff->format('Y-m-d');
}
