\
<?php
declare(strict_types=1);
require_once __DIR__ . '/db.php';

function get_categories(): array {
    $pdo = db();
    return $pdo->query("SELECT id, name FROM categories ORDER BY name COLLATE NOCASE")->fetchAll();
}

function add_category(string $name): void {
    $pdo = db();
    $stmt = $pdo->prepare("INSERT OR IGNORE INTO categories(name) VALUES(:name)");
    $stmt->execute([':name'=>$name]);
}

function delete_category(int $id): void {
    $pdo = db();
    $stmt = $pdo->prepare("DELETE FROM categories WHERE id = :id");
    $stmt->execute([':id'=>$id]);
}

function add_transaction(array $t): void {
    $pdo = db();
    $stmt = $pdo->prepare("
        INSERT INTO transactions(date, type, source, category_id, amount, merchant, note)
        VALUES(:date,:type,:source,:category_id,:amount,:merchant,:note)
    ");
    $stmt->execute([
        ':date'=>$t['date'],
        ':type'=>$t['type'],
        ':source'=>$t['source'] ?? null,
        ':category_id'=>$t['category_id'] ?? null,
        ':amount'=>$t['amount'],
        ':merchant'=>$t['merchant'] ?? null,
        ':note'=>$t['note'] ?? null,
    ]);
}

function delete_transaction(int $id): void {
    $pdo = db();
    $stmt = $pdo->prepare("DELETE FROM transactions WHERE id = :id");
    $stmt->execute([':id'=>$id]);
}

function list_transactions(array $filter): array {
    $pdo = db();
    $where = [];
    $params = [];

    if (!empty($filter['type'])) {
        $where[] = "t.type = :type";
        $params[':type'] = $filter['type'];
    }
    if (!empty($filter['date_from'])) {
        $where[] = "t.date >= :df";
        $params[':df'] = $filter['date_from'];
    }
    if (!empty($filter['date_to'])) {
        $where[] = "t.date < :dt";
        $params[':dt'] = $filter['date_to'];
    }
    if (!empty($filter['category_ids']) && is_array($filter['category_ids'])) {
        $in = [];
        foreach ($filter['category_ids'] as $i=>$cid) {
            $key = ':c'.$i;
            $in[] = $key;
            $params[$key] = (int)$cid;
        }
        if ($in) $where[] = "t.category_id IN (" . implode(',', $in) . ")";
    }

    $sql = "
        SELECT t.*, c.name AS category_name
        FROM transactions t
        LEFT JOIN categories c ON c.id = t.category_id
    ";
    if ($where) $sql .= " WHERE " . implode(" AND ", $where);
    $sql .= " ORDER BY t.date DESC, t.id DESC LIMIT 500";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

function totals(array $filter): array {
    $pdo = db();
    $where = [];
    $params = [];
    if (!empty($filter['date_from'])) { $where[]="date >= :df"; $params[':df']=$filter['date_from']; }
    if (!empty($filter['date_to'])) { $where[]="date < :dt"; $params[':dt']=$filter['date_to']; }

    $w = $where ? ("WHERE " . implode(" AND ", $where)) : "";

    $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM transactions $w AND type='income'");
    // If $w is empty, above becomes "FROM transactions  AND ..." invalid. We'll handle separately:
    if ($w === "") {
        $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM transactions WHERE type='income'");
    }
    $stmt->execute($params);
    $income = (float)$stmt->fetch()['s'];

    if ($w === "") {
        $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM transactions WHERE type='expense'");
    } else {
        $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM transactions $w AND type='expense'");
    }
    $stmt->execute($params);
    $expense = (float)$stmt->fetch()['s'];

    return ['income'=>$income, 'expense'=>$expense, 'balance'=>$income-$expense];
}

function expense_breakdown_by_category(array $filter): array {
    $pdo = db();
    $where = ["t.type='expense'"];
    $params = [];
    if (!empty($filter['date_from'])) { $where[]="t.date >= :df"; $params[':df']=$filter['date_from']; }
    if (!empty($filter['date_to'])) { $where[]="t.date < :dt"; $params[':dt']=$filter['date_to']; }
    if (!empty($filter['category_ids']) && is_array($filter['category_ids'])) {
        $in = [];
        foreach ($filter['category_ids'] as $i=>$cid) {
            $key=':c'.$i;
            $in[]=$key;
            $params[$key]=(int)$cid;
        }
        if ($in) $where[]="t.category_id IN (" . implode(',', $in) . ")";
    }
    $sql = "
        SELECT COALESCE(c.name,'(Bez kategorie)') AS category, COALESCE(SUM(t.amount),0) AS total
        FROM transactions t
        LEFT JOIN categories c ON c.id = t.category_id
        WHERE " . implode(" AND ", $where) . "
        GROUP BY category
        ORDER BY total DESC
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

function add_loan(array $l): void {
    $pdo = db();
    $stmt = $pdo->prepare("
        INSERT INTO loans(name, principal_total, start_date, planned_monthly_payment, note)
        VALUES(:name,:p,:sd,:m,:note)
    ");
    $stmt->execute([
        ':name'=>$l['name'],
        ':p'=>$l['principal_total'],
        ':sd'=>$l['start_date'],
        ':m'=>$l['planned_monthly_payment'] ?? null,
        ':note'=>$l['note'] ?? null,
    ]);
}

function delete_loan(int $id): void {
    $pdo = db();
    $stmt = $pdo->prepare("DELETE FROM loans WHERE id=:id");
    $stmt->execute([':id'=>$id]);
}

function list_loans(): array {
    $pdo = db();
    $loans = $pdo->query("SELECT * FROM loans ORDER BY created_at DESC, id DESC")->fetchAll();

    $stmtPaid = $pdo->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM loan_payments WHERE loan_id = :id");
    foreach ($loans as &$loan) {
        $stmtPaid->execute([':id'=>$loan['id']]);
        $paid = (float)$stmtPaid->fetch()['s'];
        $loan['paid'] = $paid;
        $loan['remaining'] = max(0.0, (float)$loan['principal_total'] - $paid);
    }
    return $loans;
}

function get_loan(int $id): ?array {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM loans WHERE id=:id");
    $stmt->execute([':id'=>$id]);
    $loan = $stmt->fetch();
    if (!$loan) return null;

    $stmt2 = $pdo->prepare("SELECT * FROM loan_payments WHERE loan_id=:id ORDER BY date DESC, id DESC");
    $stmt2->execute([':id'=>$id]);
    $payments = $stmt2->fetchAll();

    $paid = 0.0;
    foreach ($payments as $p) $paid += (float)$p['amount'];
    $loan['payments'] = $payments;
    $loan['paid'] = $paid;
    $loan['remaining'] = max(0.0, (float)$loan['principal_total'] - $paid);
    return $loan;
}

function add_loan_payment(array $p): void {
    $pdo = db();
    $stmt = $pdo->prepare("
        INSERT INTO loan_payments(loan_id, date, amount, note)
        VALUES(:loan_id,:date,:amount,:note)
    ");
    $stmt->execute([
        ':loan_id'=>$p['loan_id'],
        ':date'=>$p['date'],
        ':amount'=>$p['amount'],
        ':note'=>$p['note'] ?? null,
    ]);
}

function delete_loan_payment(int $id): void {
    $pdo = db();
    $stmt = $pdo->prepare("DELETE FROM loan_payments WHERE id=:id");
    $stmt->execute([':id'=>$id]);
}
