<?php
// Simple admin UI for notices. Protect this folder with Basic Auth (.htaccess) or hosting admin password protection.
?><!doctype html>
<html lang="cs">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Admin – Upozornění</title>
  <meta name="robots" content="noindex,nofollow" />
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;700;900&display=swap" rel="stylesheet">
  <style>
    :root{ --bg:#E8F6FE; --ink:#1F2938; --teal:#5AA1B7; --stroke: rgba(31,41,56,.10);
      --card: rgba(255,255,255,.86); --shadow: 0 12px 30px rgba(31,41,56,.10); --radius:18px; --warn:#FEB72E; }
    *{box-sizing:border-box}
    body{
      margin:0; font-family:"Nunito",system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif; color:var(--ink);
      background: radial-gradient(900px 500px at 20% 35%, rgba(143,212,232,.35), transparent 60%),
                  radial-gradient(700px 420px at 80% 25%, rgba(245,216,97,.35), transparent 60%),
                  linear-gradient(180deg, var(--bg), #ffffff 70%);
      min-height:100vh;
    }
    .wrap{max-width:980px;margin:0 auto;padding:18px 16px 46px;}
    header{
      background: linear-gradient(135deg, rgba(255,255,255,.90), rgba(255,255,255,.60));
      border:1px solid var(--stroke); box-shadow:var(--shadow); border-radius:var(--radius);
      padding:14px 16px; display:flex; align-items:center; justify-content:space-between; gap:12px;
    }
    .brand{display:flex; align-items:center; gap:12px; min-width:0;}
    .logo{height:44px; width:auto; display:block; filter: drop-shadow(0 6px 18px rgba(31,41,56,.10));}
    .title{font-weight:900; font-size:18px; margin:0; white-space:nowrap; overflow:hidden; text-overflow:ellipsis;}
    .sub{opacity:.75; font-weight:800; font-size:13px; margin:2px 0 0;}
    .btn{
      border:0; border-radius:999px; padding:10px 14px; font-weight:900; cursor:pointer;
      background: linear-gradient(135deg, var(--teal), rgba(143,212,232,.95));
      color:#fff; box-shadow: 0 12px 24px rgba(90,161,183,.20); text-decoration:none; display:inline-flex; gap:8px; align-items:center;
    }
    .btn.secondary{ background: rgba(255,255,255,.72); color:var(--ink); border:1px solid var(--stroke); box-shadow:none; }
    .grid{margin-top:14px; display:grid; gap:14px; grid-template-columns: 1fr 1fr;}
    @media (max-width:900px){ .grid{grid-template-columns:1fr;} }
    .card{background:var(--card); border:1px solid var(--stroke); border-radius:var(--radius); box-shadow:var(--shadow); overflow:hidden;}
    .card .inner{padding:16px;}
    h2{margin:0 0 10px; font-size:16px; letter-spacing:.2px;}
    label{display:block; font-weight:900; font-size:12px; opacity:.80; text-transform:uppercase; letter-spacing:.35px; margin:10px 0 6px;}
    input, textarea{
      width:100%; padding:10px 12px; border-radius:12px; border:1px solid var(--stroke); font:inherit; background:rgba(255,255,255,.92);
      outline:none;
    }
    textarea{min-height:96px; resize:vertical;}
    .row{display:flex; gap:10px; flex-wrap:wrap; margin-top:12px;}
    .note{margin:10px 0 0; font-size:13px; opacity:.75; font-weight:800; line-height:1.35;}
    .list{list-style:none; margin:0; padding:0; display:grid; gap:10px;}
    .item{
      background: rgba(255,255,255,.92);
      border:1px solid var(--stroke);
      border-radius:14px;
      padding:12px 12px;
      display:flex; align-items:flex-start; justify-content:space-between; gap:10px;
    }
    .item strong{display:block; margin:0; line-height:1.35;}
    .item small{display:block; margin-top:6px; opacity:.70; font-weight:800;}
    .actions{display:flex; gap:8px; flex-shrink:0;}
    .mini{
      border-radius:999px; border:1px solid var(--stroke); padding:8px 10px; font-weight:900; cursor:pointer;
      background: rgba(255,255,255,.75);
    }
    .mini.danger{border-color: rgba(254,183,46,.55);}
    .toast{margin-top:12px; font-weight:900; opacity:.85;}
    .kbd{font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace; font-size:12px; background:rgba(143,212,232,.18); padding:2px 6px; border-radius:6px; border:1px solid var(--stroke);}
    .pill{display:inline-block; padding:6px 10px; border-radius:999px; border:1px solid var(--stroke); background:rgba(255,255,255,.72); font-weight:900; font-size:12px;}
    .pill.warn{border-color: rgba(254,183,46,.55);}
  </style>
</head>
<body>
  <div class="wrap">
    <header>
      <div class="brand">
        <img class="logo" src="../assets/logo.png" alt="logo">
        <div style="min-width:0">
          <p class="title">Admin – Upozornění</p>
          <p class="sub">Změny se ukládají na server do <span class="kbd">/data/notices.json</span></p>
        </div>
      </div>
      <div class="row" style="margin:0">
        <a class="btn secondary" href="../index.html">← Zpět na web</a>
      </div>
    </header>

    <div class="grid">
      <section class="card">
        <div class="inner">
          <h2>Přidat / upravit upozornění</h2>

          <label for="text">Text upozornění</label>
          <textarea id="text" placeholder="Např. Z důvodu školení bude ordinace otevřena pouze do 12:00."></textarea>

          <label for="meta">Poznámka (volitelné)</label>
          <input id="meta" placeholder="Např. Aktualizováno: 28. 12. 2025" />

          <div class="row">
            <button class="btn" id="saveBtn">💾 Uložit</button>
            <button class="btn secondary" id="clearBtn">🧹 Vyčistit</button>
          </div>

          <p class="note">
            Klikni na <span class="kbd">Upravit</span> u položky vpravo – načte se do formuláře.
          </p>

          <div class="row" style="margin-top:14px;">
            <span class="pill">Max. 20 položek</span>
            <span class="pill warn">Po úpravě se web aktualizuje po obnovení stránky</span>
          </div>

          <div class="toast" id="toast" aria-live="polite"></div>
        </div>
      </section>

      <aside class="card">
        <div class="inner">
          <h2>Seznam upozornění</h2>
          <ul class="list" id="list"></ul>
          <p class="note">Bezpečnost: složku <span class="kbd">/admin/</span> si chraň heslem (viz README v ZIPu).</p>
        </div>
      </aside>
    </div>
  </div>

<script>
  const $ = (s) => document.querySelector(s);
  let editIndex = null;

  function toast(msg){
    $("#toast").textContent = msg;
    setTimeout(()=> { if($("#toast").textContent === msg) $("#toast").textContent = ""; }, 1800);
  }

  async function loadNotices(){
    const res = await fetch("../data/notices.json", { cache: "no-store" });
    if(!res.ok) throw new Error("Nelze načíst notices.json");
    const arr = await res.json();
    if(!Array.isArray(arr)) return [];
    return arr
      .filter(x => x && typeof x.text === "string")
      .map(x => ({ text: String(x.text).trim(), meta: x.meta ? String(x.meta).trim() : "" }))
      .filter(x => x.text.length > 0)
      .slice(0, 20);
  }

  async function saveNotices(list){
    const res = await fetch("save.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ notices: list })
    });
    const data = await res.json().catch(()=> ({}));
    if(!res.ok || !data.ok){
      throw new Error(data.error || "Uložení selhalo");
    }
    return true;
  }

  function escapeHtml(str){
    return String(str)
      .replaceAll("&","&amp;")
      .replaceAll("<","&lt;")
      .replaceAll(">","&gt;")
      .replaceAll('"',"&quot;");
  }

  async function render(){
    const list = await loadNotices();
    const ul = $("#list");
    ul.innerHTML = "";
    list.forEach((n, idx) => {
      const li = document.createElement("li");
      li.className = "item";
      li.innerHTML = `
        <div>
          <strong>${escapeHtml(n.text)}</strong>
          ${n.meta ? `<small>${escapeHtml(n.meta)}</small>` : ""}
        </div>
        <div class="actions">
          <button class="mini" data-act="edit">Upravit</button>
          <button class="mini danger" data-act="del">Smazat</button>
        </div>
      `;

      li.querySelector('[data-act="edit"]').addEventListener("click", () => {
        editIndex = idx;
        $("#text").value = n.text;
        $("#meta").value = n.meta || "";
        toast("Načteno do formuláře – uprav a ulož.");
        $("#text").focus();
      });

      li.querySelector('[data-act="del"]').addEventListener("click", async () => {
        if(!confirm("Opravdu smazat tuto položku?")) return;
        const next = await loadNotices();
        next.splice(idx, 1);
        await saveNotices(next);
        if(editIndex === idx){ editIndex = null; $("#text").value=""; $("#meta").value=""; }
        await render();
        toast("Položka smazána.");
      });

      ul.appendChild(li);
    });
  }

  $("#saveBtn").addEventListener("click", async () => {
    const text = $("#text").value.trim();
    const meta = $("#meta").value.trim();
    if(!text){
      toast("Text upozornění je povinný.");
      $("#text").focus();
      return;
    }
    const list = await loadNotices();
    const item = { text, meta };

    if(editIndex === null){
      list.unshift(item);
      toast("Upozornění přidáno.");
    }else{
      list[editIndex] = item;
      toast("Upozornění upraveno.");
      editIndex = null;
    }

    await saveNotices(list);
    $("#text").value = "";
    $("#meta").value = "";
    await render();
  });

  $("#clearBtn").addEventListener("click", () => {
    editIndex = null;
    $("#text").value = "";
    $("#meta").value = "";
    toast("Formulář vyčištěn.");
  });

  render().catch(e => toast(e.message));
</script>
</body>
</html>
