<?php
session_start();

/**
 * Jednoduchá "stejná" hra: 5 puzzlů.
 * Progres drží v $_SESSION (bez DB).
 *
 * Puzzle 1 odpovídá screenshotu: IFMMP XPSME => HELLO WORLD (Caesar +1 na cipheru, tedy -1 pro odpověď)
 */

$puzzles = [
  [
    "hint" => "A common greeting",
    "cipher" => "IFMMP XPSME",
    "answer" => "HELLO WORLD",
    "letters" => ["E","F","I","M","P","S","X"],
  ],
  [
    "hint" => "A polite phrase",
    "cipher" => "UIBOL ZPV",
    "answer" => "THANK YOU",
    "letters" => ["A","B","H","I","K","N","O","U","Y"],
  ],
  [
    "hint" => "A friendly question",
    "cipher" => "IPX BSF ZPV",
    "answer" => "HOW ARE YOU",
    "letters" => ["A","E","F","H","I","O","P","R","U","W","X","Y","Z"],
  ],
  [
    "hint" => "A simple wish",
    "cipher" => "HPPE MV DLVD",
    "answer" => "GOOD LUCK",
    "letters" => ["C","D","E","G","H","K","L","M","O","P","U","V"],
  ],
  [
    "hint" => "A common sign-off",
    "cipher" => "TFF ZPV TPP O",
    "answer" => "SEE YOU SOON",
    "letters" => ["E","F","N","O","P","S","T","U","V","Y","Z"],
  ],
];

if (!isset($_SESSION["puzzleIndex"])) $_SESSION["puzzleIndex"] = 0;
if (!isset($_SESSION["hintUsed"])) $_SESSION["hintUsed"] = 0;

if ($_SERVER["REQUEST_METHOD"] === "POST") {
  header("Content-Type: application/json; charset=utf-8");
  $action = $_POST["action"] ?? "";

  if ($action === "state") {
    $i = $_SESSION["puzzleIndex"];
    $i = max(0, min($i, count($puzzles)-1));
    $p = $puzzles[$i];

    echo json_encode([
      "ok" => true,
      "index" => $i,
      "total" => count($puzzles),
      "hint" => $p["hint"],
      "cipher" => $p["cipher"],
      "answerLen" => strlen($p["answer"]),
      "mask" => preg_replace('/[A-Z]/', '_', strtoupper($p["answer"])),
      "letters" => $p["letters"],
      "hintLeft" => max(0, 3 - $_SESSION["hintUsed"]),
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  if ($action === "reset") {
    $_SESSION["puzzleIndex"] = 0;
    $_SESSION["hintUsed"] = 0;
    echo json_encode(["ok" => true]);
    exit;
  }

  if ($action === "useHint") {
    if ($_SESSION["hintUsed"] < 3) $_SESSION["hintUsed"] += 1;
    echo json_encode(["ok" => true, "hintLeft" => max(0, 3 - $_SESSION["hintUsed"])]);
    exit;
  }

  if ($action === "check") {
    $guess = strtoupper(trim($_POST["guess"] ?? ""));
    $i = $_SESSION["puzzleIndex"];
    $p = $puzzles[$i];
    $answer = strtoupper($p["answer"]);

    $guessNorm = preg_replace('/\s+/', ' ', $guess);
    $answerNorm = preg_replace('/\s+/', ' ', $answer);

    if ($guessNorm === $answerNorm) {
      if ($i < count($puzzles) - 1) {
        $_SESSION["puzzleIndex"] += 1;
        $_SESSION["hintUsed"] = 0;
        echo json_encode(["ok" => true, "correct" => true, "finished" => false]);
      } else {
        echo json_encode(["ok" => true, "correct" => true, "finished" => true]);
      }
    } else {
      echo json_encode(["ok" => true, "correct" => false]);
    }
    exit;
  }

  echo json_encode(["ok" => false, "error" => "Unknown action"]);
  exit;
}
?>
<!doctype html>
<html lang="cs">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Crypto Puzzle</title>
  <style>
    :root{
      --bg:#0b2540;
      --primary:#0b63ff;
      --hintbg:#fdeed6;
      --hinttext:#e67e22;
      --accent:#f59e0b;
      --shadow: 0 20px 60px rgba(0,0,0,.35);
      --font: ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Arial, "Apple Color Emoji","Segoe UI Emoji";
    }
    *{box-sizing:border-box}
    body{
      margin:0;
      font-family:var(--font);
      background: radial-gradient(1200px 600px at 50% 0%, #103a63 0%, var(--bg) 55%, #061a2c 100%);
      min-height:100vh;
      display:flex;
      align-items:center;
      justify-content:center;
      padding:24px;
      color:#111827;
    }
    .phone{
      width: 360px;
      max-width: 95vw;
      border-radius: 34px;
      background:#0b0b0b;
      padding: 14px;
      box-shadow: var(--shadow);
      position:relative;
    }
    .screen{
      background: #fff;
      border-radius: 26px;
      overflow:hidden;
      min-height: 680px;
      position:relative;
    }
    .statusbar{
      height: 44px;
      display:flex;
      align-items:center;
      justify-content:center;
      font-weight:600;
      color:#111;
      opacity:.9;
    }
    .content{ padding: 10px 18px 18px; }
    .topTitle{
      text-align:center;
      color:#555;
      font-weight:600;
      margin: 8px 0 10px;
      font-size: 14px;
    }
    .hintBox{
      background: var(--hintbg);
      border-radius: 10px;
      padding: 12px 12px;
      margin: 10px 0 18px;
    }
    .hintBox .label{
      color: var(--hinttext);
      font-weight:700;
      font-size: 13px;
      margin-bottom: 6px;
    }
    .hintBox .text{ color:#2b2b2b; font-size: 14px; }
    .centerText{
      text-align:center;
      color:#6b7280;
      margin: 10px 0 10px;
      font-size: 13px;
      font-weight:600;
    }
    .answerCard{
      background:#f3f4f6;
      border-radius: 12px;
      padding: 18px 14px;
      margin: 10px 0 10px;
      display:flex;
      align-items:center;
      justify-content:center;
      min-height: 74px;
    }
    .answer{
      font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace;
      letter-spacing: 2px;
      font-size: 20px;
      color:#111;
      user-select:none;
      white-space:pre;
    }
    .cipher{
      text-align:center;
      font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace;
      color:#9ca3af;
      letter-spacing: 2px;
      margin: 6px 0 16px;
      font-size: 13px;
    }
    .keys{
      display:flex;
      flex-wrap:wrap;
      gap: 12px;
      justify-content:center;
      padding: 6px 0 10px;
    }
    .key{
      width: 74px;
      height: 64px;
      background: var(--primary);
      border:none;
      border-radius: 10px;
      color:#fff;
      font-size: 18px;
      font-weight:800;
      cursor:pointer;
      box-shadow: 0 10px 18px rgba(11,99,255,.18);
      transition: transform .06s ease, filter .15s ease;
    }
    .key:active{ transform: translateY(1px); filter: brightness(.95); }
    .controls{
      display:flex;
      gap: 12px;
      margin-top: 16px;
      align-items:center;
    }
    .btn{
      flex:1;
      height: 44px;
      border-radius: 10px;
      border:none;
      cursor:pointer;
      font-weight:800;
      display:flex;
      align-items:center;
      justify-content:center;
      gap:8px;
      font-size: 14px;
    }
    .btnHint{ background: var(--accent); color:#111; }
    .btnSolve{
      background: #e5e7eb;
      color:#9ca3af;
      cursor:not-allowed;
    }
    .btnSolve.enabled{ color:#111; cursor:pointer; }
    .subControls{
      display:flex;
      justify-content:space-between;
      margin-top: 10px;
      gap: 10px;
    }
    .smallBtn{
      flex:1;
      height: 36px;
      border-radius: 10px;
      border: 1px solid #e5e7eb;
      background:#fff;
      cursor:pointer;
      font-weight:700;
      color:#111827;
    }
    .toast{
      position:absolute;
      left: 18px;
      right: 18px;
      bottom: 18px;
      padding: 12px 12px;
      border-radius: 12px;
      background: rgba(17,24,39,.92);
      color:#fff;
      font-size: 13px;
      display:none;
    }
    .toast.show{ display:block; }
  </style>
</head>
<body>
  <div class="phone">
    <div class="screen">
      <div class="statusbar">9:41</div>

      <div class="content">
        <div class="topTitle" id="puzzleTitle">Puzzle 1 of 5</div>

        <div class="hintBox">
          <div class="label">Hint:</div>
          <div class="text" id="hintText">…</div>
        </div>

        <div class="centerText">Decrypt the phrase:</div>

        <div class="answerCard">
          <div class="answer" id="answerMask">_____ _____</div>
        </div>

        <div class="cipher" id="cipherText">IFMMP XPSME</div>

        <div class="keys" id="keys"></div>

        <div class="controls">
          <button class="btn btnHint" id="hintBtn" type="button">💡 Hint (0)</button>
          <button class="btn btnSolve" id="solveBtn" type="button" disabled>✓ Solve</button>
        </div>

        <div class="subControls">
          <button class="smallBtn" id="backBtn" type="button">⌫ Back</button>
          <button class="smallBtn" id="clearBtn" type="button">⟲ Clear</button>
          <button class="smallBtn" id="resetBtn" type="button">↺ Reset</button>
        </div>
      </div>

      <div class="toast" id="toast"></div>
    </div>
  </div>

<script>
  const els = {
    puzzleTitle: document.getElementById('puzzleTitle'),
    hintText: document.getElementById('hintText'),
    answerMask: document.getElementById('answerMask'),
    cipherText: document.getElementById('cipherText'),
    keys: document.getElementById('keys'),
    hintBtn: document.getElementById('hintBtn'),
    solveBtn: document.getElementById('solveBtn'),
    backBtn: document.getElementById('backBtn'),
    clearBtn: document.getElementById('clearBtn'),
    resetBtn: document.getElementById('resetBtn'),
    toast: document.getElementById('toast'),
  };

  let mask = "";
  let answer = "";   // pouze pro hint odhalování (demo); kontrola správnosti je server-side
  let cipher = "";
  let hintLeft = 0;

  function caesarMinus1(str){
    return str.replace(/[A-Z]/g, ch => {
      const code = ch.charCodeAt(0) - 65;
      const newCode = (code + 25) % 26;
      return String.fromCharCode(newCode + 65);
    });
  }

  function showToast(msg){
    els.toast.textContent = msg;
    els.toast.classList.add('show');
    setTimeout(()=>els.toast.classList.remove('show'), 1500);
  }

  function post(action, data = {}){
    const fd = new FormData();
    fd.append("action", action);
    for (const [k,v] of Object.entries(data)) fd.append(k, v);
    return fetch("", { method:"POST", body: fd }).then(r => r.json());
  }

  function rebuildKeys(letters){
    els.keys.innerHTML = "";
    letters.forEach(l => {
      const b = document.createElement("button");
      b.className = "key";
      b.type = "button";
      b.textContent = l;
      b.addEventListener("click", () => addLetter(l));
      els.keys.appendChild(b);
    });
  }

  function updateSolveEnabled(){
    const filled = !mask.includes("_");
    els.solveBtn.disabled = !filled;
    if (filled) els.solveBtn.classList.add("enabled");
    else els.solveBtn.classList.remove("enabled");
  }

  function addLetter(letter){
    const idx = mask.indexOf("_");
    if (idx === -1) return;
    mask = mask.slice(0, idx) + letter + mask.slice(idx+1);
    els.answerMask.textContent = mask;
    updateSolveEnabled();
  }

  function backspace(){
    for (let i = mask.length - 1; i >= 0; i--){
      if (/[A-Z]/.test(mask[i])) {
        mask = mask.slice(0, i) + "_" + mask.slice(i+1);
        break;
      }
    }
    els.answerMask.textContent = mask;
    updateSolveEnabled();
  }

  function clearAll(){
    mask = mask.replace(/[A-Z]/g, "_");
    els.answerMask.textContent = mask;
    updateSolveEnabled();
  }

  function revealOneCorrectLetter(){
    let i = mask.indexOf("_");
    if (i === -1) return false;
    while (i < mask.length && mask[i] === " "){
      i = mask.indexOf("_", i+1);
      if (i === -1) return false;
    }
    mask = mask.slice(0, i) + answer[i] + mask.slice(i+1);
    els.answerMask.textContent = mask;
    updateSolveEnabled();
    return true;
  }

  async function loadState(){
    const st = await post("state");
    if (!st.ok) return;

    els.puzzleTitle.textContent = `Puzzle ${st.index + 1} of ${st.total}`;
    els.hintText.textContent = st.hint;
    els.cipherText.textContent = st.cipher;

    cipher = st.cipher;
    answer = caesarMinus1(cipher); // demo

    mask = st.mask;
    els.answerMask.textContent = mask;

    hintLeft = st.hintLeft;
    const used = 3 - hintLeft;
    els.hintBtn.textContent = `💡 Hint (${used})`;

    rebuildKeys(st.letters);
    updateSolveEnabled();
  }

  els.hintBtn.addEventListener("click", async () => {
    if (hintLeft <= 0) { showToast("No hints left."); return; }
    const res = await post("useHint");
    if (!res.ok) return;

    hintLeft = res.hintLeft;
    const used = 3 - hintLeft;
    els.hintBtn.textContent = `💡 Hint (${used})`;

    const ok = revealOneCorrectLetter();
    if (!ok) showToast("Nothing to reveal.");
  });

  els.solveBtn.addEventListener("click", async () => {
    if (els.solveBtn.disabled) return;

    const res = await post("check", { guess: mask });
    if (!res.ok) return;

    if (!res.correct){
      showToast("Not quite. Try again!");
      return;
    }

    if (res.finished){
      showToast("✅ All puzzles solved!");
      return;
    }

    showToast("✅ Correct! Next puzzle…");
    await loadState();
  });

  els.backBtn.addEventListener("click", backspace);
  els.clearBtn.addEventListener("click", clearAll);
  els.resetBtn.addEventListener("click", async () => {
    await post("reset");
    await loadState();
    showToast("Reset done.");
  });

  loadState();
</script>
</body>
</html>
