<?php
// prices.php
// Načítá ceny (CZK) a ukládá je do cache 1× za 24 hodin.
// - CoinGecko: BTC, DOGE, WAVES, BTCZ, KICK, SWEAT, EGG
// - CoinPaprika: FONE (zdarma, bez API key), ticker id: fone-fone
//
// Vrací JSON: { lastUpdated: ISO, prices: { BTC: 123, ... } }

header('Content-Type: application/json; charset=utf-8');

$force = isset($_GET['force']) && $_GET['force'] == '1';

$cacheDir = __DIR__ . '/cache';
$cacheFile = $cacheDir . '/prices.json';
$maxAgeSeconds = 24 * 60 * 60; // 24h

// CoinGecko IDs (symbol => id)
$cg = [
  "BTC"   => "bitcoin",
  "DOGE"  => "dogecoin",
  "WAVES" => "waves",
  "BTCZ"  => "bitcoinz",
  "KICK"  => "kick",
  "SWEAT" => "sweat-economy",
  "EGG"   => "waves-ducks"
];

// CoinPaprika coin_id pro FONE
$paprikaFoneId = "fone-fone";

function now_iso() {
  return gmdate('c'); // ISO 8601 UTC
}

function read_cache($cacheFile) {
  if (!file_exists($cacheFile)) return null;
  $raw = @file_get_contents($cacheFile);
  if ($raw === false) return null;
  $data = json_decode($raw, true);
  return $data;
}

function write_cache($cacheDir, $cacheFile, $data) {
  if (!is_dir($cacheDir)) {
    @mkdir($cacheDir, 0775, true);
  }
  $raw = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  @file_put_contents($cacheFile, $raw);
}

function cache_is_fresh($cacheFile, $maxAgeSeconds) {
  if (!file_exists($cacheFile)) return false;
  $age = time() - filemtime($cacheFile);
  return $age >= 0 && $age <= $maxAgeSeconds;
}

function http_get_json($url) {
  $ctx = stream_context_create([
    'http' => [
      'method' => 'GET',
      'timeout' => 12,
      'header' => "User-Agent: CryptoPortfolioCZK/1.0\r\n"
    ]
  ]);
  $raw = @file_get_contents($url, false, $ctx);
  if ($raw === false) return null;
  $data = json_decode($raw, true);
  if (!is_array($data)) return null;
  return $data;
}

if (!$force && cache_is_fresh($cacheFile, $maxAgeSeconds)) {
  $cached = read_cache($cacheFile);
  if ($cached && isset($cached['prices'])) {
    echo json_encode($cached, JSON_UNESCAPED_UNICODE);
    exit;
  }
}

// 1) CoinGecko bulk
$cgIds = implode(',', array_values($cg));
$cgUrl = 'https://api.coingecko.com/api/v3/simple/price?' . http_build_query([
  'ids' => $cgIds,
  'vs_currencies' => 'czk'
]);

$prices = [];

$cgData = http_get_json($cgUrl);
if (is_array($cgData)) {
  foreach ($cg as $sym => $id) {
    if (isset($cgData[$id]) && isset($cgData[$id]['czk']) && is_numeric($cgData[$id]['czk'])) {
      $prices[$sym] = (float)$cgData[$id]['czk'];
    }
  }
}

// 2) CoinPaprika for FONE in CZK
// Endpoint: /v1/tickers/{coin_id}?quotes=CZK
$paprikaUrl = 'https://api.coinpaprika.com/v1/tickers/' . rawurlencode($paprikaFoneId) . '?' . http_build_query([
  'quotes' => 'CZK'
]);

$paprikaData = http_get_json($paprikaUrl);
if (is_array($paprikaData) && isset($paprikaData['quotes']['CZK']['price']) && is_numeric($paprikaData['quotes']['CZK']['price'])) {
  $prices['FONE'] = (float)$paprikaData['quotes']['CZK']['price'];
}

// Pokud se nepovedlo nic, fallback na cache (když existuje)
if (count($prices) === 0) {
  $cached = read_cache($cacheFile);
  if ($cached && isset($cached['prices'])) {
    echo json_encode($cached, JSON_UNESCAPED_UNICODE);
    exit;
  }
  http_response_code(502);
  echo json_encode([ 'error' => 'Nepodařilo se stáhnout data (CoinGecko/CoinPaprika).' ], JSON_UNESCAPED_UNICODE);
  exit;
}

$out = [
  'lastUpdated' => now_iso(),
  'prices' => $prices
];

write_cache($cacheDir, $cacheFile, $out);
echo json_encode($out, JSON_UNESCAPED_UNICODE);
