// --- Konfigurace coinů ---
const COINS = [
  { symbol: "BTC", name: "Bitcoin", auto: true },
  { symbol: "FONE", name: "Fone Network (FONE)", auto: true }, // z CoinPaprika (CZK)
  { symbol: "KICK", name: "Kick", auto: true },
  { symbol: "DOGE", name: "Dogecoin", auto: true },
  { symbol: "BTCZ", name: "BitcoinZ", auto: true },
  { symbol: "WAVES", name: "Waves", auto: true },
  { symbol: "SWEAT", name: "Sweat Economy", auto: true },
  { symbol: "EGG", name: "Waves Ducks (EGG)", auto: true },
];

const LS_AMOUNTS = "czk-portfolio-php-amounts-v1";

const elRows = document.getElementById("rows");
const elTotal = document.getElementById("total");
const elLastUpdated = document.getElementById("lastUpdated");
const elStatus = document.getElementById("status");
const elErrorBox = document.getElementById("errorBox");
const elRefreshBtn = document.getElementById("refreshBtn");

let pricesCzk = {}; // { BTC: 123, ... }
let lastUpdatedIso = null;

function parseNumber(str) {
  if (str == null) return NaN;
  return Number(String(str).trim().replace(",", "."));
}

function fmtCzk(n) {
  if (n == null || Number.isNaN(n)) return "–";
  const maxFrac = n >= 1 ? 2 : 8;
  return new Intl.NumberFormat("cs-CZ", {
    style: "currency",
    currency: "CZK",
    maximumFractionDigits: maxFrac,
  }).format(n);
}

function fmtDate(iso) {
  if (!iso) return "–";
  const d = new Date(iso);
  return new Intl.DateTimeFormat("cs-CZ", { dateStyle: "medium", timeStyle: "medium" }).format(d);
}

function getState() {
  let amounts = {};
  try {
    const a = localStorage.getItem(LS_AMOUNTS);
    if (a) amounts = JSON.parse(a);
  } catch {}

  for (const c of COINS) {
    if (!(c.symbol in amounts)) amounts[c.symbol] = "";
  }
  return { amounts };
}

function setState(amounts) {
  try { localStorage.setItem(LS_AMOUNTS, JSON.stringify(amounts)); } catch {}
}

function setError(msg) {
  if (!msg) {
    elErrorBox.classList.add("hidden");
    elErrorBox.textContent = "";
    return;
  }
  elErrorBox.textContent = msg;
  elErrorBox.classList.remove("hidden");
}

function setStatus(s) {
  elStatus.textContent = s || "";
}

// Render UI rows
function render() {
  const { amounts } = getState();

  elRows.innerHTML = "";
  let total = 0;

  for (const c of COINS) {
    const amtRaw = amounts[c.symbol] ?? "";
    const amt = parseNumber(amtRaw);

    const price = c.cgId ? pricesCzk[c.symbol] : parseNumber(manual[c.symbol] ?? "");
    const value = Number.isFinite(amt) && Number.isFinite(price) ? amt * price : null;
    if (value != null) total += value;

    const row = document.createElement("div");
    row.className = "row";

    // coin
    const coin = document.createElement("div");
    coin.className = "coin";
    coin.innerHTML = `<div class="sym">${c.symbol}</div><div class="name">${c.name}</div>`;

    // amount
    const amtCell = document.createElement("div");
    amtCell.className = "right";
    const amtInput = document.createElement("input");
    amtInput.className = "amount";
    amtInput.inputMode = "decimal";
    amtInput.placeholder = "0";
    amtInput.value = amtRaw;
    amtInput.addEventListener("input", (e) => {
      const st = getState();
      st.amounts[c.symbol] = e.target.value;
      setState(st.amounts);
      render();
    });
    amtCell.appendChild(amtInput);

    // price
    const priceCell = document.createElement("div");
    priceCell.className = "right price";
    priceCell.textContent = fmtCzk(price);

    // value
    const valueCell = document.createElement("div");
    valueCell.className = "right value";
    valueCell.textContent = fmtCzk(value);

    row.appendChild(coin);
    row.appendChild(amtCell);
    row.appendChild(priceCell);
    row.appendChild(valueCell);
    elRows.appendChild(row);
  }

  elTotal.textContent = fmtCzk(total);
  elLastUpdated.textContent = fmtDate(lastUpdatedIso);
}

// Fetch prices from PHP (server-side cache daily)
async function fetchPrices(force = false) {
  setError(null);
  setStatus("…");
  elRefreshBtn.disabled = true;

  try {
    const url = "prices.php" + (force ? "?force=1" : "");
    const res = await fetch(url, { cache: "no-store" });
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    const data = await res.json();

    if (!data || !data.prices) throw new Error("Neplatná odpověď serveru.");

    pricesCzk = data.prices;
    lastUpdatedIso = data.lastUpdated || null;

    setStatus("OK");
  } catch (e) {
    setStatus("Chyba");
    setError((e && e.message ? e.message : "Nepodařilo se načíst ceny.") +
      " (Pozn.: může jít o limit nebo problém se serverem.)");
  } finally {
    elRefreshBtn.disabled = false;
    render();
  }
}

elRefreshBtn.addEventListener("click", () => fetchPrices(true));

// Init
render();
fetchPrices(false);
