document.addEventListener('DOMContentLoaded', () => {
  const addCryptoForm = document.getElementById('add-crypto-form');
  const cryptoNameSelect = document.getElementById('crypto-name');
  const portfolioList = document.getElementById('portfolio-list');
  const portfolioSummary = document.getElementById('portfolio-summary');
  const lastUpdatedEl = document.getElementById('last-updated');
  const refreshBtn = document.getElementById('refresh-btn');
  const errorBox = document.getElementById('error-box');

  // Fixní seznam – netaháme obří CoinGecko coins list
  const SUPPORTED = [
    { symbol: 'BTC',  label: 'Bitcoin (BTC)' },
    { symbol: 'FONE', label: 'Fone Network (FONE)' },
    { symbol: 'KICK', label: 'Kick (KICK)' },
    { symbol: 'DOGE', label: 'Dogecoin (DOGE)' },
    { symbol: 'BTCZ', label: 'BitcoinZ (BTCZ)' },
    { symbol: 'WAVES',label: 'Waves (WAVES)' },
    { symbol: 'SWEAT',label: 'Sweat Economy (SWEAT)' },
    { symbol: 'EGG',  label: 'Waves Ducks (EGG)' },
  ];

  let portfolio = JSON.parse(localStorage.getItem('cryptoPortfolioCZK')) || []; // [{symbol, amount}]
  let prices = {}; // { BTC: 12345, ... } CZK
  let lastUpdatedIso = null;

  function setError(msg) {
    if (!msg) {
      errorBox.classList.add('hidden');
      errorBox.textContent = '';
      return;
    }
    errorBox.textContent = msg;
    errorBox.classList.remove('hidden');
  }

  function fmtDate(iso) {
    if (!iso) return '–';
    const d = new Date(iso);
    return new Intl.DateTimeFormat('cs-CZ', { dateStyle: 'medium', timeStyle: 'medium' }).format(d);
  }

  function fmtCzk(n) {
    if (n == null || Number.isNaN(n)) return '–';
    const maxFrac = n >= 1 ? 2 : 8;
    return new Intl.NumberFormat('cs-CZ', {
      style: 'currency',
      currency: 'CZK',
      maximumFractionDigits: maxFrac
    }).format(n);
  }

  function populateCryptoList() {
    cryptoNameSelect.innerHTML = '';
    for (const c of SUPPORTED) {
      const option = document.createElement('option');
      option.value = c.symbol;
      option.textContent = c.label;
      cryptoNameSelect.appendChild(option);
    }
  }

  addCryptoForm.addEventListener('submit', (e) => {
    e.preventDefault();
    const symbol = cryptoNameSelect.value;
    const amount = parseFloat(document.getElementById('crypto-amount').value);

    if (!symbol || !Number.isFinite(amount) || amount <= 0) return;

    const existing = portfolio.find(p => p.symbol === symbol);
    if (existing) existing.amount += amount;
    else portfolio.push({ symbol, amount });

    localStorage.setItem('cryptoPortfolioCZK', JSON.stringify(portfolio));
    addCryptoForm.reset();
    updatePortfolioUI();
  });

  function removeItem(symbol) {
    portfolio = portfolio.filter(p => p.symbol !== symbol);
    localStorage.setItem('cryptoPortfolioCZK', JSON.stringify(portfolio));
    updatePortfolioUI();
  }

  async function fetchPrices(force = false) {
    setError(null);
    refreshBtn.disabled = true;
    refreshBtn.textContent = 'Aktualizuji…';

    try {
      const url = 'prices.php' + (force ? '?force=1' : '');
      const res = await fetch(url, { cache: 'no-store' });
      if (!res.ok) throw new Error(`Server vrátil HTTP ${res.status}`);
      const data = await res.json();
      if (!data || !data.prices) throw new Error('Neplatná odpověď z prices.php');

      prices = data.prices || {};
      lastUpdatedIso = data.lastUpdated || null;
      lastUpdatedEl.textContent = fmtDate(lastUpdatedIso);
    } catch (err) {
      setError((err?.message || 'Failed to fetch') + ' — zkontroluj, že běží PHP a že existuje prices.php.');
    } finally {
      refreshBtn.disabled = false;
      refreshBtn.textContent = 'Aktualizovat teď';
      updatePortfolioUI();
    }
  }

  refreshBtn.addEventListener('click', () => fetchPrices(true));

  function updatePortfolioUI() {
    if (!portfolio.length) {
      portfolioList.innerHTML = '<p>Tvoje portfolio je prázdné.</p>';
      portfolioSummary.innerHTML = '';
      return;
    }

    let total = 0;
    portfolioList.innerHTML = '';

    for (const item of portfolio) {
      const price = prices[item.symbol];
      const value = (Number.isFinite(price) ? item.amount * price : NaN);
      if (Number.isFinite(value)) total += value;

      const row = document.createElement('div');
      row.className = 'portfolio-item';

      const left = document.createElement('div');
      left.className = 'crypto-info';
      const title = document.createElement('div');
      title.innerHTML = `
        <strong>${item.symbol}</strong><br>
        <span>${item.amount} @ ${fmtCzk(price)}</span>
      `;
      left.appendChild(title);

      const right = document.createElement('div');
      right.innerHTML = `<strong>${fmtCzk(value)}</strong>`;

      const del = document.createElement('button');
      del.type = 'button';
      del.className = 'mini-btn';
      del.textContent = 'Smazat';
      del.style.color = '#111827';
      del.style.background = '#e5e7eb';
      del.style.borderColor = '#d1d5db';
      del.addEventListener('click', () => removeItem(item.symbol));

      row.appendChild(left);
      row.appendChild(right);
      row.appendChild(del);
      portfolioList.appendChild(row);
    }

    portfolioSummary.innerHTML = `Celková hodnota portfolia: ${fmtCzk(total)}`;
  }

  populateCryptoList();
  updatePortfolioUI();
  fetchPrices(false);
});
