<?php
// Simple JSON storage helpers for prototype

function data_path(string $file): string {
  return __DIR__ . '/data/' . $file;
}

function read_json(string $file): array {
  $path = data_path($file);
  if (!file_exists($path)) return [];
  $raw = file_get_contents($path);
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}

function write_json(string $file, array $data): void {
  $path = data_path($file);
  $tmp = $path . '.tmp';
  file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
  rename($tmp, $path);
}

function h(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function now_date(): string {
  return date('Y-m-d');
}

function uuid(string $prefix): string {
  // Not a real UUID, fine for prototype
  return $prefix . bin2hex(random_bytes(4)) . dechex(time());
}

function type_label(string $type): string {
  $map = [
    'stamps' => 'Známky',
    'coins' => 'Mince',
    'banknotes' => 'Bankovky',
    'tcg' => 'TCG',
    'crypto' => 'Krypto',
    'mixed' => 'Mixed'
  ];
  return $map[$type] ?? $type;
}

function money_fmt($n, string $currency='CZK'): string {
  if ($n === null || $n === '') return '—';
  $n = (float)$n;
  $s = number_format($n, 0, ',', ' ');
  return $s . ' ' . $currency;
}

function collection_stats(array $collections, array $items): array {
  $byCollection = [];
  foreach ($collections as $c) {
    $byCollection[$c['id']] = [
      'items' => 0,
      'value' => 0,
      'missing_photo' => 0,
      'missing_value' => 0,
      'currency' => $c['currency'] ?? 'CZK'
    ];
  }
  foreach ($items as $it) {
    $cid = $it['collection_id'] ?? '';
    if (!isset($byCollection[$cid])) continue;
    $byCollection[$cid]['items'] += (int)($it['qty'] ?? 1);
    $byCollection[$cid]['value'] += (float)($it['est_value'] ?? 0);
    if (empty($it['photo'])) $byCollection[$cid]['missing_photo'] += 1;
    if (($it['est_value'] ?? null) === null || $it['est_value'] === '') $byCollection[$cid]['missing_value'] += 1;
  }
  return $byCollection;
}

function find_by_id(array $arr, string $id): ?array {
  foreach ($arr as $x) if (($x['id'] ?? '') === $id) return $x;
  return null;
}

function upsert_item(array $items, array $item): array {
  $found = false;
  foreach ($items as $i => $it) {
    if (($it['id'] ?? '') === ($item['id'] ?? '')) {
      $items[$i] = $item;
      $found = true;
      break;
    }
  }
  if (!$found) $items[] = $item;
  return $items;
}

function redirect(string $url): void {
  header('Location: ' . $url);
  exit;
}

function flash_set(string $msg, string $kind='success'): void {
  $_SESSION['flash'] = ['msg' => $msg, 'kind' => $kind];
}

function flash_get(): ?array {
  if (!isset($_SESSION['flash'])) return null;
  $f = $_SESSION['flash'];
  unset($_SESSION['flash']);
  return $f;
}
